<?php
require __DIR__ . '/inc/auth.php';
require __DIR__ . '/inc/db.php';
date_default_timezone_set('Asia/Jakarta');

$title = "Dashboard";

// ========== STATISTIK DASAR ========== //

// Total user
$total_user = 0;
if ($res = $conn->query("SELECT COUNT(*) AS jml FROM users")) {
  $row = $res->fetch_assoc();
  $total_user = (int)($row['jml'] ?? 0);
}

// Total siswa
$total_siswa = 0;
if ($res = $conn->query("SELECT COUNT(*) AS jml FROM students")) {
  $row = $res->fetch_assoc();
  $total_siswa = (int)($row['jml'] ?? 0);
}

// Total kelas
$total_kelas = 0;
if ($res = $conn->query("SELECT COUNT(*) AS jml FROM classes")) {
  $row = $res->fetch_assoc();
  $total_kelas = (int)($row['jml'] ?? 0);
}

// Tanggal hari ini
$tgl_hari_ini = date('Y-m-d');

// Total absensi (semua status) hari ini
$total_absen = 0;
if ($res = $conn->query("SELECT COUNT(*) AS jml FROM attendance WHERE tanggal = '{$tgl_hari_ini}'")) {
  $row = $res->fetch_assoc();
  $total_absen = (int)($row['jml'] ?? 0);
}

// Per status absensi hari ini
$status_data = ['Hadir' => 0, 'Izin' => 0, 'Sakit' => 0, 'Telat' => 0];
$sqlStatus = "
  SELECT status, COUNT(*) AS jml 
  FROM attendance 
  WHERE tanggal = '{$tgl_hari_ini}'
  GROUP BY status
";
if ($res = $conn->query($sqlStatus)) {
  while ($r = $res->fetch_assoc()) {
    $s = $r['status'];
    if (isset($status_data[$s])) {
      $status_data[$s] = (int)$r['jml'];
    }
  }
}

// QR aktif
$qr = 0;
if ($res = $conn->query("
  SELECT COUNT(*) AS aktif 
  FROM qr_tokens 
  WHERE expires_at IS NULL OR expires_at > NOW()
")) {
  $row = $res->fetch_assoc();
  $qr = (int)($row['aktif'] ?? 0);
}

// Rata-rata kehadiran hari ini (Hadir vs total siswa)
$hadir_hari_ini = 0;
if ($res = $conn->query("
  SELECT COUNT(*) AS jml 
  FROM attendance 
  WHERE tanggal = '{$tgl_hari_ini}' AND status = 'Hadir'
")) {
  $row = $res->fetch_assoc();
  $hadir_hari_ini = (int)($row['jml'] ?? 0);
}
$persen_hadir_hari_ini = 0.0;
if ($total_siswa > 0) {
  $persen_hadir_hari_ini = round(($hadir_hari_ini / $total_siswa) * 100, 1);
}

// ========== GRAFIK ABSEN HARIAN (X HARI TERAKHIR) ========== //

$range = isset($_GET['range']) ? (int)$_GET['range'] : 7;
if (!in_array($range, [7, 30, 60, 90], true)) {
  $range = 7;
}

$absen_chart = [];
$stmt = $conn->prepare("
  SELECT tanggal, COUNT(*) AS total 
  FROM attendance_daily
  WHERE tanggal >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
  GROUP BY tanggal
  ORDER BY tanggal ASC
");
if ($stmt) {
  $stmt->bind_param('i', $range);
  $stmt->execute();
  $result = $stmt->get_result();
  while ($row = $result->fetch_assoc()) {
    // simpan apa adanya: tanggal + total baris per hari
    $absen_chart[] = [
      'tanggal' => $row['tanggal'],
      'total'   => (int)$row['total']
    ];
  }
  $stmt->close();
}


ob_start();
?>

<!-- 🌀 SPINNER OVERLAY (HALAMAN UTAMA) -->
<div id="spinnerOverlay" class="fixed inset-0 bg-[#0b1120] flex flex-col items-center justify-center z-50">
  <div class="loader mb-4"></div>
  <p class="text-gray-400 font-medium">Memuat Dashboard...</p>
</div>

<style>
@keyframes spin { to { transform: rotate(360deg); } }

/* Loader umum */
.loader {
  border: 4px solid rgba(255, 255, 255, 0.15);
  border-top-color: #06b6d4;
  border-radius: 50%;
  width: 50px;
  height: 50px;
  animation: spin 0.8s linear infinite;
}

/* Overlay download Excel */
#downloadOverlay {
  transition: opacity 0.4s ease;
}

#downloadOverlay.hidden {
  opacity: 0;
  pointer-events: none;
}
</style>

<div class="dashboard-root p-6 rounded-2xl bg-[#141D2F] border border-white/10 text-[#E2E8F0]">
  <h1 class="text-3xl font-semibold mb-6 flex items-center gap-2">
    📊 Dashboard Analitik Kehadiran
  </h1>

  <!-- ========== KARTU STATISTIK ATAS ========== -->
  <div class="grid grid-cols-2 md:grid-cols-3 xl:grid-cols-6 gap-4 mb-6">
    <div class="bg-[#1B2845] p-4 rounded-xl shadow hover:bg-[#25345A] transition">
      <h2 class="text-[#92A2C2] text-xs">Total User</h2>
      <p class="text-2xl font-bold text-[#8CE0FF]"><?= $total_user ?></p>
    </div>

    <div class="bg-[#1B2845] p-4 rounded-xl shadow hover:bg-[#25345A] transition">
      <h2 class="text-[#92A2C2] text-xs">Total Siswa</h2>
      <p class="text-2xl font-bold text-[#8CE0FF]"><?= $total_siswa ?></p>
    </div>

    <div class="bg-[#1B2845] p-4 rounded-xl shadow hover:bg-[#25345A] transition">
      <h2 class="text-[#92A2C2] text-xs">Total Kelas</h2>
      <p class="text-2xl font-bold text-[#8CE0FF]"><?= $total_kelas ?></p>
    </div>

    <div class="bg-[#1B2845] p-4 rounded-xl shadow hover:bg-[#25345A] transition">
      <h2 class="text-[#92A2C2] text-xs">Absen Hari Ini (Semua Status)</h2>
      <p class="text-2xl font-bold text-[#8CE0FF]"><?= $total_absen ?></p>
      <p class="text-[11px] text-[#64748b] mt-1">
        Hadir: <?= $status_data['Hadir'] ?> • Izin: <?= $status_data['Izin'] ?> • Sakit: <?= $status_data['Sakit'] ?> • Telat: <?= $status_data['Telat'] ?>
      </p>
    </div>

    <div class="bg-[#1B2845] p-4 rounded-xl shadow hover:bg-[#25345A] transition">
      <h2 class="text-[#92A2C2] text-xs">Rata-rata Kehadiran Hari Ini</h2>
      <p class="text-2xl font-bold text-[#22C55E]"><?= number_format($persen_hadir_hari_ini, 1) ?>%</p>
      <p class="text-[11px] text-[#64748b] mt-1">
        <?= $hadir_hari_ini ?> siswa hadir dari <?= $total_siswa ?> siswa
      </p>
    </div>
  </div>

  <!-- ========== GRAFIK ABSEN HARIAN & RINGKASAN CEPAT ========== -->
  <div class="grid grid-cols-1 xl:grid-cols-3 gap-4 mb-6">
    <!-- Grafik Harian -->
    <div class="xl:col-span-2 bg-[#1B2845] p-5 rounded-2xl border border-white/10">
      <div class="flex flex-wrap justify-between items-center mb-3 gap-2">
  <h2 id="grafikTitle" class="text-xl font-semibold">
    Grafik Absensi <?= $range ?> Hari Terakhir
  </h2>
  <form class="flex items-center gap-2 text-sm" onsubmit="return false;">
    <label class="text-[#92A2C2]">Rentang:</label>
    <select name="range" id="rangeSelect" class="bg-[#0E1525] border border-white/10 text-white rounded px-3 py-1 cursor-pointer text-xs">
      <option value="7"  <?= $range === 7  ? 'selected' : '' ?>>7 Hari</option>
      <option value="30" <?= $range === 30 ? 'selected' : '' ?>>30 Hari</option>
      <option value="60" <?= $range === 60 ? 'selected' : '' ?>>60 Hari</option>
      <option value="90" <?= $range === 90 ? 'selected' : '' ?>>90 Hari</option>
    </select>
  </form>
</div>

      <canvas id="absenChart" height="100"></canvas>
    </div>

    <!-- Ringkasan Cepat (periode rekap di bawah) -->
    <div class="bg-[#1B2845] p-5 rounded-2xl border border-white/10 flex flex-col justify-between gap-3">
      <div>
        <p class="text-sm text-[#92A2C2] mb-1">📅 Tanggal Hari Ini</p>
        <p class="text-xl font-semibold"><?= date('d M Y') ?></p>
      </div>
      <div>
        <p class="text-sm text-[#92A2C2] mb-1">📊 Rata-rata Kehadiran Semua Kelas (Periode Terpilih)</p>
        <p id="avgAll" class="text-3xl font-bold text-cyan-400">0.0%</p>
      </div>
      <div>
        <p id="trendInfo" class="text-xs text-[#cbd5f5] bg-[#0f172a]/60 border border-white/10 rounded-lg px-3 py-2">
          Tren vs periode sebelumnya: -
        </p>
      </div>
    </div>
  </div>

  <!-- ========== PANEL REKAP STATISTIK KELAS (VERSI LAMA + UPGRADE) ========== -->

  <div class="w-full max-w-7xl mx-auto bg-[#141d2f] shadow-xl rounded-2xl p-5 border border-white/10">
    <div class="flex flex-wrap justify-between items-center gap-3 mb-4">
      <h2 class="text-lg font-semibold text-cyan-300 flex items-center gap-2">
        📊 Rekap Statistik Kehadiran Kelas
      </h2>
    </div>

    <!-- Filter Periode + Filter Tingkat + Export -->
    <div class="flex flex-wrap gap-3 items-center mb-4">
      <label class="text-sm">Periode:</label>
      <select id="periode" class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1 text-sm">
        <option value="2025-01-01,2025-06-30">Jan–Jun 2025</option>
        <option value="2025-07-01,2025-12-31" selected>Jul–Des 2025</option>
      </select>

      <button id="btnTampil" class="bg-green-600 hover:bg-green-700 text-white px-4 py-1.5 rounded-lg text-sm">
        ✅ Tampilkan
      </button>

      <div class="flex items-center gap-2 ml-auto">
        <label class="text-sm">Filter Kelas:</label>
        <select id="filterTingkat" class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1 text-sm">
          <option value="semua">Semua kelas</option>
        </select>

        <button id="btnExport" class="bg-blue-500 hover:bg-blue-600 px-3 py-1.5 rounded-lg text-white text-sm">
          📤 Export Excel
        </button>
      </div>
    </div>

    <!-- Insight Top / Bottom / Rata-rata -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
      <div class="bg-[#1B2845] p-4 rounded-xl border border-white/10">
        <p class="text-xs text-[#92A2C2] mb-1">📈 Kelas Rata-rata Tertinggi</p>
        <p id="topKelas" class="text-sm font-semibold text-[#22C55E]">-</p>
      </div>
      <div class="bg-[#1B2845] p-4 rounded-xl border border-white/10">
        <p class="text-xs text-[#92A2C2] mb-1">📉 Kelas Rata-rata Terendah</p>
        <p id="bottomKelas" class="text-sm font-semibold text-[#EF4444]">-</p>
      </div>
      <div class="bg-[#1B2845] p-4 rounded-xl border border-white/10">
        <p class="text-xs text-[#92A2C2] mb-1">📊 Rata-rata Semua Kelas (Filter Saat Ini)</p>
        <p class="text-sm text-slate-300">
          Nilai persentase ditampilkan di kartu kanan atas dan di tabel di bawah.
        </p>
      </div>
    </div>

    <!-- Tabel Rekap -->
    <div class="overflow-x-auto mb-5 rounded-lg border border-white/10">
      <table class="min-w-full text-xs text-center" id="rekapTable">
        <thead id="tableHeader" class="bg-[#243B55] text-white"></thead>
        <tbody id="tableBody" class="bg-[#0b1120]"></tbody>
      </table>
    </div>

    <!-- Grafik Per Kelas -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
      <div class="bg-[#1a253d] p-3 rounded-lg border border-white/10">
        <h3 class="text-sm font-semibold text-gray-300 mb-2">
          📈 Tren Persentase Kehadiran (Line Chart)
        </h3>
        <canvas id="lineChart" height="200"></canvas>
      </div>
      <div class="bg-[#1a253d] p-3 rounded-lg border border-white/10">
        <h3 class="text-sm font-semibold text-gray-300 mb-2">
          📊 Perbandingan Antar Kelas (Bar Chart)
        </h3>
        <canvas id="barChart" height="200"></canvas>
      </div>
    </div>
  </div>
</div>

<!-- 🔄 Overlay Download Excel -->
<div id="downloadOverlay"
     class="hidden fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 text-center">
  <div class="flex flex-col items-center justify-center bg-[#141d2f]/90 rounded-2xl px-8 py-6 border border-white/10 shadow-2xl">
    <div class="loader mb-4 scale-110"></div>
    <p class="text-gray-200 font-medium text-base tracking-wide">Menyiapkan file Excel...</p>
  </div>
</div>

<!-- Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script>
// ========== GRAFIK ABSEN HARIAN DINAMIS (TANPA RELOAD) ==========
let absenChartInstance = null;

function renderAbsenChart(labels, values) {
  const ctx = document.getElementById('absenChart');
  if (!ctx) return;

  if (absenChartInstance) {
    absenChartInstance.destroy();
  }

  absenChartInstance = new Chart(ctx, {
    type: 'line',
    data: {
      labels,
      datasets: [{
        label: 'Total Absen',
        data: values,
        borderColor: '#8CE0FF',
        backgroundColor: '#5B8CFF33',
        tension: 0.4,
        fill: true
      }]
    },
    options: {
      scales: {
        y: {
          beginAtZero: true,
          ticks: { color: '#94a3b8', precision: 0 }
        },
        x: {
          ticks: { color: '#94a3b8' }
        }
      },
      plugins: {
        legend: { display: false }
      }
    }
  });
}

document.addEventListener('DOMContentLoaded', () => {
  // Data awal dari PHP (range default)
  const initialLabels = <?= json_encode(array_column($absen_chart, 'tanggal')) ?>;
  const initialValues = <?= json_encode(array_map('intval', array_column($absen_chart, 'total'))) ?>;

  if (initialLabels.length) {
    renderAbsenChart(initialLabels, initialValues);
  }

  const rangeSelect = document.getElementById('rangeSelect');
  const titleEl     = document.getElementById('grafikTitle');

  if (rangeSelect) {
    rangeSelect.addEventListener('change', async (e) => {
      const range = e.target.value;

      try {
        const res = await fetch(`/system-absent/inc/api_chart_harian.php?range=${range}`);
        const json = await res.json();
        if (!json.success) {
          console.error(json.error || 'Gagal memuat data grafik harian');
          return;
        }

        renderAbsenChart(json.labels, json.values);

        if (titleEl) {
          titleEl.textContent = `Grafik Absensi ${json.range} Hari Terakhir`;
        }
      } catch (err) {
        console.error('Error fetch grafik harian:', err);
      }
    });
  }
});


// ========== REKAP STATISTIK KELAS (FETCH API) ==========

const REKAP_API_URL    = '/system-absent/inc/api_rekap_dashboard.php';
const REKAP_EXPORT_URL = '/system-absent/public/export_rekap_statistik_excel.php';

let lineChart = null;
let barChart  = null;
let rekapDataGlobal = {};
let currentMonths = [];
let filterOptionsInitialized = false;

function monthName(id) {
  const map = ['Jan','Feb','Mar','Apr','Mei','Jun','Jul','Agu','Sep','Okt','Nov','Des'];
  return map[id - 1] || '-';
}

function getMonthRange(start, end) {
  const s = new Date(start);
  const e = new Date(end);
  const months = [];
  if (isNaN(s) || isNaN(e)) return months;

  while (s <= e) {
    months.push(s.getMonth() + 1);
    s.setMonth(s.getMonth() + 1);
  }
  return months;
}

function renderTableHeader(months) {
  let header = `<tr>
    <th class="p-2 border bg-[#243B55] text-white text-center">KELAS</th>`;
  months.forEach(m => {
    header += `
      <th class="p-2 border bg-[#243B55] text-white text-center">${monthName(m)}</th>
      <th class="p-2 border bg-[#243B55] text-white text-center">${monthName(m)} (%)</th>`;
  });
  header += `
    <th class="p-2 border bg-[#243B55] text-white text-center">TOTAL HADIR</th>
    <th class="p-2 border bg-[#243B55] text-white text-center">RATA-RATA</th>
  </tr>`;
  document.getElementById("tableHeader").innerHTML = header;
}

function renderTableBody(data, months) {
  const bodyEl = document.getElementById("tableBody");
  const avgAllEl = document.getElementById("avgAll");
  const topEl = document.getElementById("topKelas");
  const bottomEl = document.getElementById("bottomKelas");

  let body = '';
  let totalPersenAll = 0;
  let classCount = 0;
  const avgMap = {};

  const kelasList = Object.keys(data);
  if (!kelasList.length) {
    bodyEl.innerHTML = `<tr><td colspan="${months.length * 2 + 3}" class="p-3 text-gray-400">Tidak ada data untuk periode ini</td></tr>`;
    if (avgAllEl)   avgAllEl.textContent = '0.0%';
    if (topEl)      topEl.textContent = '-';
    if (bottomEl)   bottomEl.textContent = '-';
    return;
  }

  for (const kelas of kelasList) {
    const bulanArr = Array.isArray(data[kelas]) ? data[kelas] : [];
    let totalHadir = 0;
    let sumPersen  = 0;
    let countPersen = 0;

    let row = `<tr class="hover:bg-[#1E2B44] transition">
      <td class="p-2 border font-semibold">${kelas}</td>`;

    months.forEach(m => {
      const bulan = bulanArr.find(b => b.bulan === m);
      const hadir = bulan ? (bulan.hadir ?? 0) : 0;
      const persen = bulan ? (bulan.persen ?? 0) : 0;
      totalHadir += hadir;
      if (bulan) {
        sumPersen += persen;
        countPersen++;
      }
      row += `
        <td class="p-2 border">${hadir}</td>
        <td class="p-2 border text-cyan-300">${persen.toFixed ? persen.toFixed(1) : persen}%</td>`;
    });

    const avg = countPersen ? (sumPersen / countPersen) : 0;
    avgMap[kelas] = avg;
    totalPersenAll += avg;
    classCount++;

    row += `
      <td class="p-2 border font-bold text-white">${totalHadir}</td>
      <td class="p-2 border text-cyan-300">${avg.toFixed(1)}%</td>
    </tr>`;

    body += row;
  }

  bodyEl.innerHTML = body;

  const avgAll = classCount ? (totalPersenAll / classCount) : 0;
  if (avgAllEl) {
    avgAllEl.textContent = avgAll.toFixed(1) + '%';
  }

  // Top & Bottom
  const entries = Object.entries(avgMap).sort((a, b) => b[1] - a[1]); // desc
  if (entries.length && topEl && bottomEl) {
    const top3 = entries.slice(0, 3)
      .map(([k, v]) => `${k} (${v.toFixed(1)}%)`)
      .join(', ');
    topEl.textContent = top3;

    const bottom = entries[entries.length - 1];
    bottomEl.textContent = `${bottom[0]} (${bottom[1].toFixed(1)}%)`;
  }
}

function renderCharts(data, months) {
  const labels = months.map(monthName);
  const datasets = Object.keys(data).map((kelas, i) => {
    const colors = ['#3b82f6', '#ef4444', '#eab308', '#22c55e', '#8b5cf6', '#06b6d4'];
    const color = colors[i % colors.length];
    const values = months.map(m => {
      const bulanData = (data[kelas] || []).find(b => b.bulan === m);
      return bulanData ? (bulanData.persen ?? 0) : 0;
    });

    return {
      label: kelas,
      borderColor: color,
      backgroundColor: color + '90',
      data: values
    };
  });

  const lineCtx = document.getElementById('lineChart');
  const barCtx  = document.getElementById('barChart');

  if (lineChart) lineChart.destroy();
  if (barChart)  barChart.destroy();

  if (lineCtx) {
    lineChart = new Chart(lineCtx, {
      type: 'line',
      data: { labels, datasets },
      options: {
        responsive: true,
        plugins: {
          legend: { labels: { color: '#e5e7eb', font: { size: 10 } } }
        },
        scales: {
          x: { ticks: { color: '#9ca3af' } },
          y: { ticks: { color: '#9ca3af' }, beginAtZero: true }
        }
      }
    });
  }

  if (barCtx) {
    barChart = new Chart(barCtx, {
      type: 'bar',
      data: { labels, datasets },
      options: {
        responsive: true,
        plugins: {
          legend: { labels: { color: '#e5e7eb', font: { size: 10 } } }
        },
        scales: {
          x: { ticks: { color: '#9ca3af' } },
          y: { ticks: { color: '#9ca3af' }, beginAtZero: true }
        }
      }
    });
  }
}

function initFilterOptions() {
  if (filterOptionsInitialized) return;
  const select = document.getElementById('filterTingkat');
  if (!select) return;

  const prefixes = new Set();
  Object.keys(rekapDataGlobal).forEach(kelas => {
    const prefix = (kelas || '').split(' ')[0];
    if (prefix) prefixes.add(prefix);
  });

  select.innerHTML = '<option value="semua">Semua kelas</option>';
  prefixes.forEach(prefix => {
    const opt = document.createElement('option');
    opt.value = prefix;
    opt.textContent = prefix;
    select.appendChild(opt);
  });
  filterOptionsInitialized = true;
}

function applyFilter() {
  const select = document.getElementById('filterTingkat');
  const filter = select ? select.value : 'semua';

  const filtered = {};
  Object.keys(rekapDataGlobal).forEach(kelas => {
    const prefix = (kelas || '').split(' ')[0];
    if (filter === 'semua' || filter === prefix) {
      filtered[kelas] = rekapDataGlobal[kelas];
    }
  });

  renderTableBody(filtered, currentMonths);
  renderCharts(filtered, currentMonths);
}

function getPreviousRange(start, end) {
  const s = new Date(start);
  const e = new Date(end);
  if (isNaN(s) || isNaN(e)) return { start, end };

  const diffMs = e.getTime() - s.getTime();
  const prevEnd = new Date(s.getTime() - 24 * 60 * 60 * 1000);
  const prevStart = new Date(prevEnd.getTime() - diffMs);

  const toStr = d => d.toISOString().slice(0, 10);
  return { start: toStr(prevStart), end: toStr(prevEnd) };
}

async function updateTrend(start, end) {
  const trendEl = document.getElementById('trendInfo');
  if (!trendEl) return;

  const { start: prevStart, end: prevEnd } = getPreviousRange(start, end);
  if (!prevStart || !prevEnd) return;

  try {
    const res = await fetch(`${REKAP_API_URL}?start=${prevStart}&end=${prevEnd}`);
    const json = await res.json();
    if (!json.success || !json.data) {
      trendEl.textContent = 'Tren vs periode sebelumnya: -';
      return;
    }

    const dataPrev = json.data;
    let totalPrev = 0;
    let countPrev = 0;

    Object.keys(dataPrev).forEach(kelas => {
      const arr = Array.isArray(dataPrev[kelas]) ? dataPrev[kelas] : [];
      if (!arr.length) return;
      const avg = arr.reduce((sum, b) => sum + (b.persen ?? 0), 0) / arr.length;
      totalPrev += avg;
      countPrev++;
    });

    if (!countPrev) {
      trendEl.textContent = 'Tren vs periode sebelumnya: -';
      return;
    }

    const prevAvg = totalPrev / countPrev;
    const currentAvgText = (document.getElementById('avgAll')?.textContent || '0').replace('%', '');
    const currentAvg = parseFloat(currentAvgText);
    if (isNaN(currentAvg)) {
      trendEl.textContent = 'Tren vs periode sebelumnya: -';
      return;
    }

    const diff = currentAvg - prevAvg;
    let arrow = '➡️';
    if (diff > 0.05) arrow = '⬆️';
    else if (diff < -0.05) arrow = '⬇️';

    trendEl.textContent = `Tren vs periode sebelumnya: ${arrow} ${diff.toFixed(1)} poin`;
  } catch (err) {
    trendEl.textContent = 'Tren vs periode sebelumnya: -';
  }
}

async function loadRekapData(updateTrendAlso = true) {
  const periodeVal = document.getElementById('periode').value.split(',');
  const [start, end] = periodeVal;
  const months = getMonthRange(start, end);
  currentMonths = months;
  renderTableHeader(months);

  try {
    const res = await fetch(`${REKAP_API_URL}?start=${start}&end=${end}`);
    const json = await res.json();
    if (!json.success) throw new Error(json.error || 'Gagal memuat data');
    rekapDataGlobal = json.data || {};
    initFilterOptions();
    applyFilter();

    if (updateTrendAlso) {
      updateTrend(start, end);
    }
  } catch (err) {
    document.getElementById("tableBody").innerHTML =
      `<tr><td colspan="20" class="text-red-400 p-3">⚠️ ${err.message}</td></tr>`;
    const avgAllEl = document.getElementById("avgAll");
    if (avgAllEl) avgAllEl.textContent = '0.0%';
  }
}

// ========== SPINNER HALAMAN UTAMA ==========

window.addEventListener("load", () => {
  const spinner = document.getElementById("spinnerOverlay");
  const content = document.querySelector(".dashboard-root");

  if (spinner && content) {
    content.style.opacity = "0";
    spinner.style.transition = "opacity 0.6s ease";
    spinner.style.opacity = "0";

    setTimeout(() => {
      spinner.remove();
      content.style.transition = "opacity 0.8s ease";
      content.style.opacity = "1";
    }, 650);
  }
});

// ========== EXPORT EXCEL ==========

document.getElementById('btnExport')?.addEventListener('click', async () => {
  const periode = document.getElementById('periode').value.split(',');
  const [start, end] = periode;

  const overlay = document.getElementById('downloadOverlay');
  overlay.classList.remove('hidden');

  try {
    const url = `${REKAP_EXPORT_URL}?start=${start}&end=${end}`;
    const res = await fetch(url);
    if (!res.ok) throw new Error('Gagal menyiapkan file.');

    const disposition = res.headers.get('Content-Disposition');
    let filename = 'Rekap_Statistik.xlsx';
    if (disposition && disposition.includes('filename=')) {
      filename = disposition.split('filename=')[1].replace(/["']/g, '');
    }

    const blob = await res.blob();
    const blobUrl = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = blobUrl;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();

    setTimeout(() => {
      window.URL.revokeObjectURL(blobUrl);
      overlay.classList.add('hidden');
    }, 800);
  } catch (err) {
    overlay.classList.add('hidden');
    alert('⚠️ Terjadi kesalahan: ' + err.message);
  }
});

// ========== EVENT & AUTO REFRESH ==========

document.getElementById('btnTampil')?.addEventListener('click', () => {
  loadRekapData(true);
});

document.getElementById('filterTingkat')?.addEventListener('change', () => {
  applyFilter();
});

// Load awal rekap saat halaman siap
window.addEventListener('DOMContentLoaded', () => {
  loadRekapData(true);
});

// Auto-refresh rekap tiap 5 menit (tanpa ganggu user)
setInterval(() => {
  loadRekapData(false);
}, 300000);
</script>

<?php
$content = ob_get_clean();
require __DIR__ . '/inc/layout.php';
