<?php
require __DIR__ . '/../../inc/db.php';
require __DIR__ . '/../../vendor/phpqrcode/qrlib.php';
date_default_timezone_set('Asia/Jakarta');
session_start();

/*
|--------------------------------------------------------------------------
| MODE API (FETCH / AJAX)
|--------------------------------------------------------------------------
| POST  → generate QR & return JSON
| GET   → tampilkan halaman HTML + preview modal
|--------------------------------------------------------------------------
*/

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  header('Content-Type: application/json');

  // ======================
  // Ambil input
  // ======================
  $class_raw  = $_POST['class_id'] ?? '';
  $event_name = trim($_POST['event_name'] ?? '');
  $start_date = $_POST['start_date'] ?? '';
  $end_date   = $_POST['end_date'] ?? '';
  $created_by = $_SESSION['user_id'] ?? 0;
  session_write_close();

  // Tentukan tipe QR
  $is_event = ($class_raw === 'event');
  $qr_type  = $is_event ? 'event' : 'rutin';
  $class_id = $is_event ? null : (int)$class_raw;

  // ======================
  // Validasi
  // ======================
  if (!$start_date || !$end_date) {
    echo json_encode(['success' => false, 'message' => '⚠️ Mohon isi semua data.']);
    exit;
  }

  if (!$is_event && $class_id <= 0) {
    echo json_encode(['success' => false, 'message' => '⚠️ Kelas tidak valid.']);
    exit;
  }

  if ($is_event && $event_name === '') {
    echo json_encode(['success' => false, 'message' => '⚠️ Nama kegiatan wajib diisi untuk EVENT.']);
    exit;
  }

  $start_time = strtotime($start_date);
  $end_time   = strtotime($end_date);

  if ($end_time <= $start_time) {
    echo json_encode(['success' => false, 'message' => '⚠️ Waktu berakhir harus lebih besar dari waktu mulai.']);
    exit;
  }

  // ======================
  // Generate token
  // ======================
  $token = bin2hex(random_bytes(16));
  $attendance_date = date('Y-m-d', $start_time);

  // ======================
  // Simpan database (STABIL)
  // ======================
  $stmt = $conn->prepare("
    INSERT INTO attendance_qr
    (class_id, event_name, qr_type, qr_token, start_date, end_date, attendance_date, created_by)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
  ");

  $stmt->bind_param(
    "issssssi",
    $class_id,
    $event_name,
    $qr_type,
    $token,
    date('Y-m-d H:i:s', $start_time),
    date('Y-m-d H:i:s', $end_time),
    $attendance_date,
    $created_by
  );

  if (!$stmt->execute()) {
    echo json_encode([
      'success' => false,
      'message' => 'DB Error: ' . $stmt->error
    ]);
    exit;
  }

  // ======================
  // Generate QR Image
  // ======================
  $qrDir = __DIR__ . '/generated_qr/';
  if (!is_dir($qrDir)) {
    mkdir($qrDir, 0775, true);
  }

  $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $domain   = $_SERVER['HTTP_HOST'];

  $qrUrl  = "$protocol://$domain/qr/$token";
  $qrFile = $qrDir . "$token.png";

  QRcode::png($qrUrl, $qrFile, QR_ECLEVEL_H, 6, 2);

  // ======================
  // JSON Response
  // ======================
  echo json_encode([
    'success'      => true,
    'qr_url'       => $qrUrl,
    'qr_img'       => "generated_qr/$token.png",
    'qr_type'      => $qr_type,
    'event_name'   => $event_name,
    'active_from'  => date('d M Y H:i', $start_time),
    'active_until' => date('d M Y H:i', $end_time)
  ]);
  exit;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="UTF-8">
  <title>Generate QR Absensi</title>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>

<body>

<div class="qrabsen-wrapper">
  <div class="form-card">
    <h1 class="text-xl text-center font-bold mb-4 text-blue-400">📅 Generate QR Absensi</h1>

    <form id="formGenerateQR" class="space-y-4">

      <!-- Nama Event (hanya EVENT) -->
      <div id="eventNameWrapper" style="display:none">
        <label>Nama Kegiatan</label>
        <input type="text" name="event_name" placeholder="Contoh: Acara A">
      </div>

      <div>
        <label>Pilih Kelas</label>
        <select name="class_id" id="classSelect" required>
          <option value="">-- Pilih Kelas --</option>
          <option value="event">📌 EVENT (Semua Kelas)</option>
          <?php
          $q = $conn->query("SELECT id, name FROM classes ORDER BY name");
          while ($r = $q->fetch_assoc()) {
            echo "<option value='{$r['id']}'>{$r['name']}</option>";
          }
          ?>
        </select>
      </div>

      <div>
        <label>Tanggal & Jam Mulai Aktif</label>
        <input type="datetime-local" name="start_date" required>
      </div>

      <div>
        <label>Tanggal & Jam Berakhir Aktif</label>
        <input type="datetime-local" name="end_date" required>
      </div>

      <button type="submit" class="btn-primary">Generate QR</button>
    </form>
  </div>
</div>

<!-- MODAL PREVIEW QR -->
<div id="qrModal">
  <div class="modal-content">
    <span class="close-btn" id="closeModal">&times;</span>
    <h2 class="text-lg font-semibold text-blue-400 mb-3">QR Absensi</h2>
    <div id="qrContainer"></div>
    <div>
      <button class="modal-btn" id="copyLinkBtn">Salin Tautan</button>
      <button class="modal-btn" id="shareWaBtn">Share WhatsApp</button>
    </div>
  </div>
</div>

<script>
const form = document.getElementById('formGenerateQR');
const classSelect = document.getElementById('classSelect');
const eventWrapper = document.getElementById('eventNameWrapper');

const qrModal = document.getElementById('qrModal');
const qrContainer = document.getElementById('qrContainer');
const closeModal = document.getElementById('closeModal');
const copyLinkBtn = document.getElementById('copyLinkBtn');
const shareWaBtn = document.getElementById('shareWaBtn');

let currentQRUrl = '';

classSelect.addEventListener('change', () => {
  eventWrapper.style.display = (classSelect.value === 'event') ? 'block' : 'none';
});

form.addEventListener('submit', async (e) => {
  e.preventDefault();
  const formData = new FormData(form);

  const res = await fetch('generate_qr.php', {
    method: 'POST',
    body: formData
  });

  const d = await res.json();

  if (!d.success) {
    alert(d.message);
    return;
  }

  currentQRUrl = d.qr_url;

  qrContainer.innerHTML = `
    <img src="${d.qr_img}">
    <p class="text-sm">Aktif: ${d.active_from} — ${d.active_until}</p>
    <a href="${d.qr_url}" target="_blank">${d.qr_url}</a>
  `;

  qrModal.classList.add('active');
});

closeModal.onclick = () => qrModal.classList.remove('active');
copyLinkBtn.onclick = async () => {
  await navigator.clipboard.writeText(currentQRUrl);
  alert('Link disalin!');
};
shareWaBtn.onclick = () => {
  window.open(`https://wa.me/?text=${encodeURIComponent(currentQRUrl)}`);
};
</script>

</body>
</html>
