<?php

require __DIR__ . '/../../inc/auth.php';
require __DIR__ . '/../../inc/db.php';
date_default_timezone_set('Asia/Jakarta');

// Ambil daftar kelas
$classes = $conn->query("SELECT id, name FROM classes ORDER BY name");

// Inisialisasi variabel untuk feedback (mode NON-AJAX)
$success           = false; // sukses generate token
$activity_success  = false; // sukses tambah master kegiatan
$error             = '';
$generatedToken    = '';
$generatedUrl      = '';
$selectedClassName = '';

function generateRandomCode($length = 4) {
    $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $max   = strlen($chars) - 1;
    $code  = '';

    for ($i = 0; $i < $length; $i++) {
        $code .= $chars[random_int(0, $max)];
    }
    return $code;
}

/**
 * Generate token 6 karakter:
 * 2 karakter pertama = "kode kelas" (dari nama kelas)
 * 4 karakter belakang = random
 */
function generateShortTokenForClass(mysqli $conn, $className) {
    // Ambil huruf/angka dari nama kelas, uppercase
    $slug   = preg_replace('/[^A-Z0-9]/', '', strtoupper($className));
    $prefix = substr($slug, 0, 2);

    // fallback kalau nama kelas terlalu pendek / kosong
    if ($prefix === '' || strlen($prefix) < 2) {
        $prefix = 'CL';
    }

    // Loop sampai dapat token yang belum dipakai
    do {
        $random = generateRandomCode(4);
        $token  = $prefix . $random; // total 6 karakter

        $esc   = $conn->real_escape_string($token);
        $check = $conn->query("SELECT 1 FROM attendance_qr WHERE qr_token = '$esc' LIMIT 1");
    } while ($check && $check->num_rows > 0);

    return $token;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $mode   = $_POST['mode'] ?? 'token';
    $isAjax = isset($_POST['ajax']) && $_POST['ajax'] === '1';

    // ==========================
    // 1) MODE: TAMBAH KEGIATAN
    // ==========================
    if ($mode === 'activity') {
        $activity_name = trim($_POST['activity_name'] ?? '');
        $jsonResponse  = ['success' => false];

        if ($activity_name === '') {
            $error = "Nama kegiatan wajib diisi.";
            $jsonResponse['error'] = $error;
        } else {
            $stmtAct = $conn->prepare("INSERT INTO activities (name, is_active) VALUES (?, 1)");
            if ($stmtAct) {
                $stmtAct->bind_param('s', $activity_name);
                if ($stmtAct->execute()) {
                    $activity_success     = true;
                    $newId                = $conn->insert_id;
                    $jsonResponse['success'] = true;
                    $jsonResponse['id']      = $newId;
                    $jsonResponse['name']    = $activity_name;
                } else {
                    $error = "Gagal menyimpan kegiatan: " . $stmtAct->error;
                    $jsonResponse['error'] = $error;
                }
            } else {
                $error = "Gagal menyiapkan query untuk kegiatan.";
                $jsonResponse['error'] = $error;
            }
        }

        if ($isAjax) {
            header('Content-Type: application/json');
            echo json_encode($jsonResponse);
            exit;
        }

    // ==========================
    // 2) MODE: HAPUS KEGIATAN
    // ==========================
    } elseif ($mode === 'delete_activity') {
        $id          = (int)($_POST['id'] ?? 0);
        $jsonResponse = ['success' => false];

        if (!$id) {
            $error = "ID kegiatan tidak valid.";
            $jsonResponse['error'] = $error;
        } else {
            // Soft delete: is_active = 0
            $stmtDel = $conn->prepare("UPDATE activities SET is_active = 0 WHERE id = ?");
            if ($stmtDel) {
                $stmtDel->bind_param('i', $id);
                if ($stmtDel->execute()) {
                    $jsonResponse['success'] = true;
                } else {
                    $error = "Gagal menghapus kegiatan: " . $stmtDel->error;
                    $jsonResponse['error'] = $error;
                }
            } else {
                $error = "Gagal menyiapkan query hapus kegiatan.";
                $jsonResponse['error'] = $error;
            }
        }

        if ($isAjax) {
            header('Content-Type: application/json');
            echo json_encode($jsonResponse);
            exit;
        }

    // ==========================
    // 3) MODE: GENERATE TOKEN
    // ==========================
    } else {
        $class_id   = (int)($_POST['class_id'] ?? 0);
        $start_date = trim($_POST['start_date'] ?? '');
        $end_date   = trim($_POST['end_date'] ?? '');
        $jsonResponse = [
            'success'    => false,
            'error'      => '',
            'token'      => null,
            'url'        => null,
            'class_name' => null,
        ];

        // Validasi sederhana
        if (!$class_id || !$start_date || !$end_date) {
            $error = "Kelas, tanggal mulai, dan tanggal akhir wajib diisi.";
            $jsonResponse['error'] = $error;
        } else {
            // Formatkan jadi datetime penuh
            $start_dt = $start_date . ' 00:00:00';
            $end_dt   = $end_date   . ' 23:59:59';

            // Cek: end_date tidak boleh < start_date
            if (strtotime($end_dt) < strtotime($start_dt)) {
                $error = "Tanggal akhir tidak boleh lebih kecil dari tanggal mulai.";
                $jsonResponse['error'] = $error;
            } else {
                // Ambil nama kelas (untuk ditampilkan & jadi prefix token)
                $clsRes = $conn->query("SELECT name FROM classes WHERE id = $class_id LIMIT 1");
                if ($clsRes && $clsRes->num_rows) {
                    $clsRow            = $clsRes->fetch_assoc();
                    $selectedClassName = $clsRow['name'];
                } else {
                    $selectedClassName = '';
                }

                // 🔑 Generate token pendek (6 karakter) per kelas
                $token = generateShortTokenForClass($conn, $selectedClassName ?: ('CLASS' . $class_id));

                // 🔖 Flag type = 'rutin'
                $type = 'rutin';

                // INSERT ke attendance_qr
                $stmt = $conn->prepare("
                    INSERT INTO attendance_qr (class_id, start_date, end_date, qr_token, qr_type, created_at)
                    VALUES (?, ?, ?, ?, ?, NOW())
                ");

                if ($stmt) {
                    $stmt->bind_param('issss', $class_id, $start_dt, $end_dt, $token, $type);
                    if ($stmt->execute()) {
                        $success        = true;
                        $generatedToken = $token;

                        // URL absensi rutin (form input token)
                        $scheme     = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                        $host       = $_SERVER['HTTP_HOST'];
                        $publicPath = '/absensi'; // sudah kamu mapping di .htaccess

                        $generatedUrl = $scheme . '://' . $host . $publicPath;

                        $jsonResponse['success']    = true;
                        $jsonResponse['token']      = $generatedToken;
                        $jsonResponse['url']        = $generatedUrl;
                        $jsonResponse['class_name'] = $selectedClassName;
                    } else {
                        $error = "Gagal menyimpan ke database: " . $stmt->error;
                        $jsonResponse['error'] = $error;
                    }
                } else {
                    $error = "Gagal menyiapkan query database.";
                    $jsonResponse['error'] = $error;
                }
            }
        }

        if ($isAjax) {
            header('Content-Type: application/json');
            echo json_encode($jsonResponse);
            exit;
        }
    }
}

// Ambil daftar kegiatan aktif (untuk tampilan awal)
$activities = $conn->query("SELECT id, name FROM activities WHERE is_active = 1 ORDER BY name");

/* ==========================
 * MULAI KONTEN UNTUK LAYOUT
 * ========================== */
$page_title = 'Generate Token Absen Rutin per Kelas';
ob_start();
?>

<!-- LOADING OVERLAY -->
<div id="pageLoader" class="fixed inset-0 z-50 flex items-center justify-center bg-slate-900/90 backdrop-blur-sm">
    <div class="flex flex-col items-center gap-3">
        <div class="w-10 h-10 border-4 border-slate-500 border-t-blue-400 rounded-full animate-spin"></div>
        <p class="text-sm text-slate-200">Memuat halaman...</p>
    </div>
</div>

<div class="max-w-4xl mx-auto px-4 py-6">
    <h1 class="text-2xl font-bold text-blue-400 mb-1 flex items-center gap-2">
        <span>🔁</span><?= htmlspecialchars($page_title) ?>
    </h1>

    <!-- Area alert dinamis (untuk AJAX) -->
    <div id="dynamicAlertArea"></div>

    <!-- Alert bawaan (kalau pernah submit non-AJAX) -->
    <?php if ($error): ?>
        <div id="mainAlert" class="mb-4 rounded-lg border border-red-500/60 bg-red-500/10 px-4 py-2 text-sm text-red-200 transition-opacity duration-500">
            ⚠️ <?= htmlspecialchars($error) ?>
        </div>
    <?php endif; ?>

    <?php if ($activity_success): ?>
        <div id="activityAlert" class="mb-4 rounded-lg border border-emerald-500/60 bg-emerald-500/10 px-4 py-2 text-sm text-emerald-200 transition-opacity duration-500">
            ✅ Jenis kegiatan baru berhasil ditambahkan.
        </div>
    <?php endif; ?>

    <!-- Hasil generate token (akan di-update via AJAX) -->
    <div id="tokenResult">
        <?php if ($success && $generatedToken): ?>
            <div class="mb-6 rounded-lg border border-emerald-500/60 bg-emerald-500/10 px-4 py-3 text-sm">
                <p class="font-semibold text-emerald-300 flex flex-wrap items-center gap-2 mb-1">
                    <span>✅ Token rutin berhasil dibuat!</span>
                    <?php if ($selectedClassName): ?>
                        <span class="text-xs text-slate-300">
                            (Kelas: <?= htmlspecialchars($selectedClassName) ?>)
                        </span>
                    <?php endif; ?>
                </p>
                <p class="text-xs text-slate-300 mb-2">
                    Berikan token ini ke guru/siswa. Akses melalui URL absensi rutin.
                </p>

                <div class="mt-2 grid gap-2">
                    <div class="text-xs flex items-center gap-2">
                        <span class="text-slate-400">Token:</span>
                        <span class="font-mono text-lg tracking-widest text-emerald-300 bg-slate-900/80 px-3 py-1 rounded-lg border border-emerald-500/40">
                            <?= htmlspecialchars($generatedToken) ?>
                        </span>
                    </div>
                    <div class="text-xs">
                        <span class="text-slate-400">URL Absensi:</span>
                        <input type="text"
                            readonly
                            class="mt-1 w-full rounded-md bg-slate-900 border border-slate-700 px-3 py-1.5 text-[11px] font-mono"
                            value="<?= htmlspecialchars($generatedUrl) ?>">
                        <p class="mt-1 text-[11px] text-slate-400">
                            Siswa cukup buka URL ini, lalu masukkan token di atas.
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- MASTER JENIS KEGIATAN -->
    <div class="rounded-2xl border border-slate-800 bg-slate-900/60 p-5 mb-6">
        <h2 class="text-lg font-semibold text-slate-100 mb-2">
            Master Jenis Kegiatan
        </h2>
        <form id="activityForm" method="POST" class="flex flex-col sm:flex-row gap-3 items-start">
            <input type="hidden" name="mode" value="activity">

            <input type="text"
                   name="activity_name"
                   placeholder="Nama kegiatan, misal: Pengajian Rutin"
                   class="flex-1 rounded-lg bg-slate-950 border border-slate-700 px-3 py-2 text-sm text-slate-100 focus:outline-none focus:ring-2 focus:ring-blue-500">

            <button type="submit"
                    class="inline-flex items-center gap-2 rounded-lg bg-emerald-600 px-4 py-2 text-sm font-semibold text-white hover:bg-emerald-500">
                <span class="activity-btn-label">➕ Tambah Kegiatan</span>
            </button>
        </form>

        <p class="mt-3 text-xs text-slate-400">
            Daftar kegiatan aktif:
        </p>
        <ul id="activityList" class="mt-1 text-xs text-slate-200 flex flex-wrap gap-2">
            <?php if ($activities && $activities->num_rows): ?>
                <?php while ($a = $activities->fetch_assoc()): ?>
                    <li class="px-2 py-1 rounded bg-slate-800 border border-slate-700 flex items-center gap-2" data-id="<?= $a['id'] ?>">
                        <span><?= htmlspecialchars($a['name']) ?></span>
                        <button type="button" class="delete-activity text-[10px] text-red-300 hover:text-red-100" title="Hapus kegiatan">
                            ✕
                        </button>
                    </li>
                <?php endwhile; ?>
            <?php else: ?>
                <li class="text-slate-500">Belum ada data kegiatan.</li>
            <?php endif; ?>
        </ul>
    </div>

    <!-- FORM GENERATE TOKEN -->
    <div class="rounded-2xl border border-slate-800 bg-slate-900/60 p-5">
        <form id="tokenForm" method="POST" class="space-y-4">
            <input type="hidden" name="mode" value="token">

            <!-- PILIH KELAS -->
            <div>
                <label class="block text-sm font-medium text-slate-200 mb-1">
                    Kelas
                </label>
                <select name="class_id"
                        class="w-full rounded-lg bg-slate-950 border border-slate-700 px-3 py-2 text-sm text-slate-100 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">-- Pilih Kelas --</option>
                    <?php if ($classes && $classes->num_rows): ?>
                        <?php while ($c = $classes->fetch_assoc()): ?>
                            <option value="<?= $c['id'] ?>">
                                <?= htmlspecialchars($c['name']) ?>
                            </option>
                        <?php endwhile; ?>
                    <?php endif; ?>
                </select>
            </div>

            <!-- TANGGAL MULAI & AKHIR (RENTANG BERLAKU) -->
            <div class="grid sm:grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-slate-200 mb-1">
                        Tanggal Mulai Berlaku
                    </label>
                    <input type="date"
                           name="start_date"
                           value="<?= htmlspecialchars($_POST['start_date'] ?? date('Y-m-d')) ?>"
                           class="w-full rounded-lg bg-slate-950 border border-slate-700 px-3 py-2 text-sm text-slate-100 focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-slate-200 mb-1">
                        Tanggal Berakhir
                    </label>
                    <input type="date"
                           name="end_date"
                           value="<?= htmlspecialchars($_POST['end_date'] ?? date('Y-m-d', strtotime('+6 months'))) ?>"
                           class="w-full rounded-lg bg-slate-950 border border-slate-700 px-3 py-2 text-sm text-slate-100 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <p class="mt-1 text-[11px] text-slate-400">
                        Misalnya 1 semester (3–6 bulan) untuk absensi rutin.
                    </p>
                </div>
            </div>

            <div class="pt-2 border-t border-slate-800 flex justify-end">
                <button type="submit"
                        class="mt-2 inline-flex items-center gap-2 rounded-lg bg-blue-600 px-4 py-2 text-sm font-semibold text-white hover:bg-blue-500">
                    🔁 Generate Token Rutin (6 Karakter)
                </button>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const pageLoader   = document.getElementById('pageLoader');
    const tokenForm    = document.getElementById('tokenForm');
    const tokenResult  = document.getElementById('tokenResult');
    const activityForm = document.getElementById('activityForm');
    const activityList = document.getElementById('activityList');
    const dynamicAlert = document.getElementById('dynamicAlertArea');

    // === helper: tampilkan alert cantik, auto-hide ===
    function showAlert(type, message) {
        if (!dynamicAlert) return;
        const div = document.createElement('div');
        const base = 'mb-4 rounded-lg px-4 py-2 text-sm transition-opacity duration-500 ';
        let classes = '';
        let icon = '⚠️ ';
        if (type === 'success') {
            classes = 'border border-emerald-500/60 bg-emerald-500/10 text-emerald-200';
            icon = '✅ ';
        } else {
            classes = 'border border-red-500/60 bg-red-500/10 text-red-200';
        }
        div.className = base + classes;
        div.textContent = icon + message;
        dynamicAlert.appendChild(div);

        setTimeout(() => { div.style.opacity = '0'; }, 3500);
        setTimeout(() => { div.remove(); }, 4000);
    }

    // Hilangkan loader setelah halaman siap
    if (pageLoader) {
        setTimeout(() => {
            pageLoader.style.opacity = '0';
            pageLoader.style.pointerEvents = 'none';
            setTimeout(() => pageLoader.style.display = 'none', 300);
        }, 300);
    }

    // Auto-hide alert PHP kalau ada
    const mainAlert = document.getElementById('mainAlert');
    if (mainAlert) {
        setTimeout(() => {
            mainAlert.style.opacity = '0';
            mainAlert.style.pointerEvents = 'none';
        }, 3500);
    }
    const activityAlert = document.getElementById('activityAlert');
    if (activityAlert) {
        setTimeout(() => {
            activityAlert.style.opacity = '0';
            activityAlert.style.pointerEvents = 'none';
        }, 3500);
    }

    // === AJAX: tambah kegiatan ===
    if (activityForm && activityList) {
        activityForm.addEventListener('submit', function (e) {
            e.preventDefault();

            const btn   = activityForm.querySelector('button[type="submit"]');
            const label = btn.querySelector('.activity-btn-label');
            const input = activityForm.querySelector('input[name="activity_name"]');
            const originalLabel = label.textContent;
            const nameVal = (input.value || '').trim();

            if (!nameVal) {
                showAlert('error', 'Nama kegiatan wajib diisi.');
                return;
            }

            const formData = new FormData(activityForm);
            formData.append('ajax', '1');

            btn.disabled = true;
            label.textContent = 'Menyimpan...';

            fetch('', { method: 'POST', body: formData })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        // kalau list sebelumnya berisi "Belum ada data kegiatan." hapus dulu
                        const emptyItem = activityList.querySelector('li.text-slate-500');
                        if (emptyItem) emptyItem.remove();

                        const li = document.createElement('li');
                        li.className = 'px-2 py-1 rounded bg-slate-800 border border-slate-700 flex items-center gap-2';
                        li.dataset.id = data.id;
                        li.innerHTML = `
                            <span>${data.name}</span>
                            <button type="button" class="delete-activity text-[10px] text-red-300 hover:text-red-100" title="Hapus kegiatan">✕</button>
                        `;
                        activityList.appendChild(li);
                        input.value = '';
                        showAlert('success', 'Kegiatan berhasil ditambahkan.');
                        bindDeleteButtons(); // re-bind delete untuk item baru
                    } else {
                        showAlert('error', data.error || 'Gagal menyimpan kegiatan.');
                    }
                })
                .catch(() => {
                    showAlert('error', 'Terjadi kesalahan koneksi.');
                })
                .finally(() => {
                    btn.disabled = false;
                    label.textContent = originalLabel;
                });
        });
    }

    // === fungsi: bind tombol hapus kegiatan ===
    function bindDeleteButtons() {
        const buttons = document.querySelectorAll('.delete-activity');
        buttons.forEach(btn => {
            if (btn.dataset.bound === '1') return; // cegah double bind
            btn.dataset.bound = '1';

            btn.addEventListener('click', function () {
                const li = btn.closest('li[data-id]');
                if (!li) return;
                const id = li.dataset.id;
                if (!id) return;

                if (!confirm('Hapus kegiatan ini?')) return;

                const fd = new FormData();
                fd.append('mode', 'delete_activity');
                fd.append('id', id);
                fd.append('ajax', '1');

                fetch('', { method: 'POST', body: fd })
                    .then(res => res.json())
                    .then(data => {
                        if (data.success) {
                            li.remove();
                            showAlert('success', 'Kegiatan berhasil dihapus.');
                        } else {
                            showAlert('error', data.error || 'Gagal menghapus kegiatan.');
                        }
                    })
                    .catch(() => {
                        showAlert('error', 'Terjadi kesalahan koneksi.');
                    });
            });
        });
    }

    bindDeleteButtons(); // panggil awal

    // === AJAX: generate token (tanpa reload) ===
    if (tokenForm && tokenResult) {
        tokenForm.addEventListener('submit', function (e) {
            e.preventDefault();

            const formData = new FormData(tokenForm);
            formData.append('ajax', '1');

            if (pageLoader) {
                pageLoader.style.display = 'flex';
                pageLoader.style.opacity = '1';
                pageLoader.style.pointerEvents = 'auto';
            }

            fetch('', { method: 'POST', body: formData })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        const kelasText = data.class_name ? ` (Kelas: ${data.class_name})` : '';
                        tokenResult.innerHTML = `
                            <div class="mb-6 rounded-lg border border-emerald-500/60 bg-emerald-500/10 px-4 py-3 text-sm">
                                <p class="font-semibold text-emerald-300 flex flex-wrap items-center gap-2 mb-1">
                                    <span>✅ Token rutin berhasil dibuat!</span>
                                    ${data.class_name ? `<span class="text-xs text-slate-300">${kelasText}</span>` : ''}
                                </p>
                                <p class="text-xs text-slate-300 mb-2">
                                    Berikan token ini ke guru/siswa. Akses melalui URL absensi rutin.
                                </p>
                                <div class="mt-2 grid gap-2">
                                    <div class="text-xs flex items-center gap-2">
                                        <span class="text-slate-400">Token:</span>
                                        <span class="font-mono text-lg tracking-widest text-emerald-300 bg-slate-900/80 px-3 py-1 rounded-lg border border-emerald-500/40">
                                            ${data.token}
                                        </span>
                                    </div>
                                    <div class="text-xs">
                                        <span class="text-slate-400">URL Absensi:</span>
                                        <input type="text"
                                            readonly
                                            class="mt-1 w-full rounded-md bg-slate-900 border border-slate-700 px-3 py-1.5 text-[11px] font-mono"
                                            value="${data.url}">
                                        <p class="mt-1 text-[11px] text-slate-400">
                                            Siswa cukup buka URL ini, lalu masukkan token di atas.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        `;
                        showAlert('success', 'Token berhasil digenerate.');
                    } else {
                        showAlert('error', data.error || 'Gagal generate token.');
                    }
                })
                .catch(() => {
                    showAlert('error', 'Terjadi kesalahan koneksi.');
                })
                .finally(() => {
                    if (pageLoader) {
                        pageLoader.style.opacity = '0';
                        pageLoader.style.pointerEvents = 'none';
                        setTimeout(() => { pageLoader.style.display = 'none'; }, 300);
                    }
                });
        });
    }
});
</script>

<?php

$content = ob_get_clean();
require __DIR__ . '/../../inc/layout.php';
