<?php
require __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

date_default_timezone_set('Asia/Jakarta');

// ============================
// 🔹 Ambil Parameter
// ============================
$token    = $_GET['token'] ?? '';
$class_id = isset($_GET['class_id']) ? (int)$_GET['class_id'] : 0;
$bulan    = isset($_GET['bulan']) ? (int)$_GET['bulan'] : 0;
$tahun    = isset($_GET['tahun']) ? (int)$_GET['tahun'] : 0;

if (!$token || !$class_id) die('❌ Parameter tidak lengkap.');

// ============================
// 🔹 Validasi Token
// ============================
$stmt = $conn->prepare("SELECT start, end, expired_at, use_period FROM rekap_public_links WHERE token = ?");
$stmt->bind_param('s', $token);
$stmt->execute();
$link = $stmt->get_result()->fetch_assoc();
if (!$link) die('❌ Token tidak valid.');
if (strtotime($link['expired_at']) < time()) die('❌ Token sudah kadaluarsa.');

// ============================
// 🔹 Tentukan Periode
// ============================
if (isset($link['use_period']) && (int)$link['use_period'] === 0) {
  // Mode tanpa periode → ambil bulan/tahun dari URL
  $bulan = $bulan ?: date('n');
  $tahun = $tahun ?: date('Y');
  $start = sprintf('%04d-%02d-01', $tahun, $bulan);
  $end   = date('Y-m-t', strtotime($start));
} else {
  // Mode periode → pakai start-end dari link
  $start = $link['start'];
  $end   = $link['end'];
}

// ============================
// 🔹 Ambil Data Per Tanggal
// ============================
$tanggalStmt = $conn->prepare("
  SELECT DISTINCT DATE(a.tanggal) AS tanggal
  FROM attendance_daily a
  INNER JOIN students s ON s.id = a.student_id
  WHERE s.class_id = ? AND a.tanggal BETWEEN ? AND ?
  ORDER BY tanggal ASC
");
$tanggalStmt->bind_param('iss', $class_id, $start, $end);
$tanggalStmt->execute();
$tanggalRes = $tanggalStmt->get_result();

// ============================
// 🔹 Buat Spreadsheet
// ============================
$spreadsheet = new Spreadsheet();
$sheet       = $spreadsheet->getActiveSheet();
$sheet->setTitle('Rekap Per Tanggal');

$row = 1;
$sheet->setCellValue("A$row", "Rekap Kehadiran Siswa (Per Tanggal)");
$sheet->getStyle("A$row")->getFont()->setBold(true)->setSize(14);
$row++;

// Info periode (otomatis adaptif)
$periodeText = (isset($link['use_period']) && (int)$link['use_period'] === 0)
  ? "Bulan: " . date('F Y', strtotime($start))
  : "Periode: {$start} s/d {$end}";
$sheet->setCellValue("A$row", $periodeText);
$row += 2;

// ============================
// 🔹 Loop Setiap Tanggal
// ============================
while ($tgl = $tanggalRes->fetch_assoc()) {
  $tanggal = $tgl['tanggal'];

  // Ambil data siswa + nama kegiatan
  $stmt2 = $conn->prepare("
    SELECT 
      s.name,
      a.status,
      act.name AS activity_name
    FROM students s
    LEFT JOIN attendance_daily a 
      ON a.student_id = s.id 
     AND DATE(a.tanggal) = ?
    LEFT JOIN activities act
      ON act.id = a.activity_id
    WHERE s.class_id = ?
    ORDER BY s.name
  ");
  $stmt2->bind_param('si', $tanggal, $class_id);
  $stmt2->execute();
  $res = $stmt2->get_result();

  $data       = [];
  $hadirCount = 0;
  $uniqueActs = [];

  while ($r = $res->fetch_assoc()) {
    if ($r['status'] === 'Hadir') {
      $hadirCount++;
    }

    // simpan ke array data (nama, status, kegiatan)
    $activityName = $r['activity_name'] ?? '';
    $data[] = [
      'name'          => $r['name'],
      'status'        => $r['status'],
      'activity_name' => $activityName,
    ];

    // kumpulkan kegiatan unik untuk header
    if ($activityName !== '' && !in_array($activityName, $uniqueActs, true)) {
      $uniqueActs[] = $activityName;
    }
  }

  $totalSiswa = count($data);

  // Tentukan label kegiatan per tanggal (sama logika dengan view publik)
  $activityLabel = '-';
  if (count($uniqueActs) === 1) {
    $activityLabel = $uniqueActs[0];
  } elseif (count($uniqueActs) > 1) {
    $activityLabel = 'Beragam kegiatan';
  }

  $tanggalLabel = strftime('%A, %d %B %Y', strtotime($tanggal));

  // Header tanggal
  $sheet->setCellValue("A{$row}", "📅 {$tanggalLabel} — {$hadirCount} hadir dari {$totalSiswa} siswa");
  $sheet->getStyle("A{$row}")->getFont()->setBold(true)->getColor()->setARGB('FF1E88E5');
  $row++;

  // Baris keterangan kegiatan (tambahan baru, tampilan tetap rapi)
  $sheet->setCellValue("A{$row}", "Kegiatan: {$activityLabel}");
  $sheet->getStyle("A{$row}")->getFont()->setItalic(true)->getColor()->setARGB('FF555555');
  $row++;

  // Satu baris kosong sebelum tabel (seperti sebelumnya)
  $row++;

  // Header tabel
  $sheet->setCellValue("A{$row}", "Nama");
  $sheet->setCellValue("B{$row}", "Status");
  $sheet->getStyle("A{$row}:B{$row}")->getFont()->setBold(true);
  $sheet->getStyle("A{$row}:B{$row}")->getAlignment()->setHorizontal('center');
  $sheet->getStyle("A{$row}:B{$row}")
        ->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
        ->getStartColor()->setARGB('FFDCE6F1');
  $row++;

  // Data siswa
  foreach ($data as $d) {
    $sheet->setCellValue("A{$row}", $d['name']);
    $sheet->setCellValue("B{$row}", $d['status'] ?: '-');
    $row++;
  }

  // Jarak antar tanggal
  $row += 2;
}

// ============================
// 🔹 Format & Styling
// ============================
foreach (range('A', 'B') as $col) {
  $sheet->getColumnDimension($col)->setAutoSize(true);
}

$lastRow = $sheet->getHighestRow();
$sheet->getStyle("A1:B{$lastRow}")
  ->getBorders()->getAllBorders()
  ->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN)
  ->getColor()->setARGB('FFBBBBBB');

// ============================
// 🔹 Output File
// ============================
$periodeSlug = (isset($link['use_period']) && (int)$link['use_period'] === 0)
  ? date('Y_m', strtotime($start))
  : "{$link['start']}_sd_{$link['end']}";

$filename = "Rekap_PerTanggal_{$periodeSlug}.xlsx";
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header("Content-Disposition: attachment; filename=\"$filename\"");
header('Cache-Control: max-age=0');

$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit;
?>
