<?php
require_once __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// ============================
// 🔹 Ambil Parameter
// ============================
$start = $_GET['start'] ?? '';
$end   = $_GET['end']   ?? '';
if (!$start || !$end) die('❌ Parameter periode tidak lengkap.');

// ============================
// 🔹 Ambil Data (SELARAS DENGAN UI)
// ============================
// Rumus sama seperti public_api/rekap_statistik.php (versi per kelas)
$stmt = $conn->prepare("
  SELECT 
    c.id   AS class_id,
    c.name AS class_name,
    MONTH(a.tanggal) AS bulan,
    COUNT(CASE WHEN a.status = 'Hadir' THEN 1 END) AS total_hadir,
    COUNT(DISTINCT DATE(a.tanggal)) AS total_kegiatan,
    (SELECT COUNT(*) FROM students s WHERE s.class_id = c.id) AS total_siswa,
    ROUND(
      (
        COUNT(CASE WHEN a.status = 'Hadir' THEN 1 END) /
        (
          (SELECT COUNT(*) FROM students s WHERE s.class_id = c.id) *
          NULLIF(COUNT(DISTINCT DATE(a.tanggal)), 0)
        )
      ) * 100, 1
    ) AS persen
  FROM attendance_daily a
  JOIN classes c ON c.id = a.class_id
  WHERE a.tanggal BETWEEN ? AND ?
  GROUP BY c.id, bulan
  ORDER BY c.name, bulan
");
$stmt->bind_param('ss', $start, $end);
$stmt->execute();
$res = $stmt->get_result();

$data = [];
while ($r = $res->fetch_assoc()) {
  $class  = $r['class_name'];
  $bulan  = (int)$r['bulan'];
  $hadir  = (int)$r['total_hadir'];
  $persen = (float)($r['persen'] ?? 0);

  $data[$class][] = [
    'bulan'  => $bulan,
    'hadir'  => $hadir,
    'persen' => $persen,
  ];
}

// ============================
// 🔹 Buat Spreadsheet
// ============================
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();
$sheet->setTitle('Rekap Statistik');

// Warna tema UI
$darkBlue = '243B55';
$navy     = '1a253d';
$cyan     = '06b6d4';
$white    = 'FFFFFF';

// ============================
// 🔹 Judul & Periode
// ============================
$row = 1;
$sheet->mergeCells("A{$row}:H{$row}");
$sheet->setCellValue("A{$row}", "📊 Rekap Statistik Kehadiran Kelas");
$sheet->getStyle("A{$row}")->getFont()->setBold(true)->setSize(14)->getColor()->setARGB("FF$cyan");
$sheet->getStyle("A{$row}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

$row++;
$sheet->mergeCells("A{$row}:H{$row}");
$sheet->setCellValue("A{$row}", "Periode: {$start} s/d {$end}");
$sheet->getStyle("A{$row}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

$row += 2;

// ============================
// 🔹 Header Tabel
// ============================
$months = [];
$startMonth = (int)date('n', strtotime($start));
$endMonth   = (int)date('n', strtotime($end));
for ($m = $startMonth; $m <= $endMonth; $m++) {
  $months[] = $m;
}

$col = 'A';
$sheet->setCellValue("{$col}{$row}", "KELAS");
$sheet->getColumnDimension($col++)->setWidth(30);

foreach ($months as $m) {
  $monthName = strftime('%b', mktime(0, 0, 0, $m, 1));
  $sheet->setCellValue("{$col}{$row}", $monthName);
  $sheet->getColumnDimension($col++)->setWidth(12);

  $sheet->setCellValue("{$col}{$row}", "{$monthName} (%)");
  $sheet->getColumnDimension($col++)->setWidth(12);
}

$sheet->setCellValue("{$col}{$row}", "TOTAL HADIR");
$sheet->getColumnDimension($col++)->setWidth(14);
$sheet->setCellValue("{$col}{$row}", "RATA-RATA");
$sheet->getColumnDimension($col)->setWidth(14);

// Style header
$sheet->getStyle("A{$row}:{$col}{$row}")->applyFromArray([
  'font' => ['bold' => true, 'color' => ['argb' => "FF$white"]],
  'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
  'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => "FF$darkBlue"]],
]);

$row++;

// ============================
// 🔹 Isi Data (Per Kelas)
// ============================
$totalSemua  = 0;
$jumlahKelas = 0;

foreach ($data as $kelas => $records) {
  $sheet->setCellValue("A{$row}", $kelas);
  $sheet->getStyle("A{$row}")->getFont()->setBold(true)->getColor()->setARGB("FF$white");

  $col         = 'B';
  $totalHadir  = 0;
  $sumPersen   = 0;
  $countPersen = 0;

  foreach ($months as $m) {
    $found  = array_values(array_filter($records, fn($r) => $r['bulan'] == $m));
    $hadir  = $found ? $found[0]['hadir']  : 0;
    $persen = $found ? $found[0]['persen'] : 0;

    $totalHadir += $hadir;
    if ($found) {
      $sumPersen   += $persen;
      $countPersen++;
    }

    $sheet->setCellValue("{$col}{$row}", $hadir);  $col++;
    $sheet->setCellValue("{$col}{$row}", "{$persen}%"); $col++;
  }

  // rata-rata per kelas (sama logika dengan UI)
  $avg = $countPersen ? round($sumPersen / $countPersen, 1) : 0;

  $sheet->setCellValue("{$col}{$row}", $totalHadir); $col++;
  $sheet->setCellValue("{$col}{$row}", "{$avg}%");
  $sheet->getStyle("{$col}{$row}")->getFont()->getColor()->setARGB("FF$cyan");

  // styling baris
  $sheet->getStyle("A{$row}:{$col}{$row}")->applyFromArray([
    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => "FF$navy"]],
    'font' => ['color' => ['argb' => "FF$white"]],
    'borders' => ['allBorders' => [
      'borderStyle' => Border::BORDER_THIN,
      'color'       => ['argb' => 'FF1E2B44']
    ]],
  ]);

  $totalSemua  += $avg;
  $jumlahKelas++;
  $row++;
}

// ============================
// 🔹 Rata-rata Semua Kelas (rata-rata antar kelas)
// ============================
$row += 1;
$sheet->mergeCells("A{$row}:{$col}{$row}");
$rataAll = $jumlahKelas ? round($totalSemua / $jumlahKelas, 1) : 0;

$sheet->setCellValue("A{$row}", "📈 Rata-rata Kehadiran Semua Kelas: {$rataAll}%");
$sheet->getStyle("A{$row}")->applyFromArray([
  'font' => ['bold' => true, 'color' => ['argb' => "FF$cyan"], 'size' => 12],
  'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
]);

// ============================
// 🔹 Output File
// ============================
$filename = "Rekap_Statistik_{$start}_sd_{$end}.xlsx";
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header("Content-Disposition: attachment; filename=\"$filename\"");
header('Cache-Control: max-age=0');

$writer = new Xlsx($spreadsheet);
$writer->save('php://output');
exit;
?>
