<?php
require_once __DIR__ . '/../inc/db.php';

// Periode dinamis berdasarkan tahun sekarang
$year     = date('Y');
$prevYear = $year - 1;
?>
<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Rekap Statistik Kehadiran Kelas</title>
  <link rel="icon" type="image/png" href="/system-absent/assets/icons/logo.png">
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    @keyframes spin { to { transform: rotate(360deg); } }

    .loader {
      border: 4px solid rgba(255, 255, 255, 0.15);
      border-top-color: #06b6d4;
      border-radius: 50%;
      width: 50px;
      height: 50px;
      animation: spin 0.8s linear infinite;
    }

    #downloadOverlay {
      transition: opacity 0.4s ease;
    }
    #downloadOverlay.hidden {
      opacity: 0;
      pointer-events: none;
    }
  </style>
</head>
<body class="bg-[#0b1120] text-gray-100 min-h-screen flex flex-col items-center p-4">

<!-- 🌀 SPINNER HALAMAN -->
<div id="spinnerOverlay" class="fixed inset-0 bg-[#020617] flex flex-col items-center justify-center z-50">
  <div class="loader mb-4"></div>
  <p class="text-gray-400 font-medium text-sm">Memuat Rekap Statistik Kehadiran...</p>
</div>

<main class="page-root w-full max-w-7xl mx-auto space-y-4">

  <!-- HEADER -->
  <header class="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
    <div>
      <h1 class="text-xl sm:text-2xl font-semibold text-cyan-300 flex items-center gap-2">
        📊 Rekap Statistik Kehadiran Kelas
      </h1>
          </div>
  </header>

  <!-- KARTU RATA-RATA & FILTER -->
  <section class="bg-[#141d2f] shadow-xl rounded-2xl p-4 sm:p-5 border border-white/10">
    <div class="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
      <div>
        <p class="text-xs text-[#94a3b8] mb-1">📊 Rata-rata Kehadiran Semua Kelas</p>
        <p id="avgAll" class="text-3xl font-bold text-cyan-400">0.0%</p>
        <p id="avgAllInfo" class="text-[11px] text-slate-400 mt-1">
          Berdasarkan seluruh kelas pada periode yang dipilih.
        </p>
      </div>

      <div class="flex flex-wrap items-center gap-2 justify-start sm:justify-end text-xs sm:text-sm">
        <div class="flex items-center gap-1">
          <label class="text-slate-300">Periode:</label>
          <select id="periode" class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1">
            <option value="<?= $year ?>-01-01,<?= $year ?>-12-31" selected>Tahun <?= $year ?></option>
            <option value="<?= $year ?>-01-01,<?= $year ?>-06-30">Jan–Jun <?= $year ?></option>
            <option value="<?= $year ?>-07-01,<?= $year ?>-12-31">Jul–Des <?= $year ?></option>
            <option value="<?= $prevYear ?>-01-01,<?= $prevYear ?>-12-31">Tahun <?= $prevYear ?></option>
          </select>
        </div>

        <div class="flex items-center gap-1">
          <label class="text-slate-300">Filter Kelas:</label>
          <select id="filterTingkat" class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1">
            <option value="semua">Semua kelas</option>
          </select>
        </div>

        <button id="btnTampil"
                class="bg-emerald-600 hover:bg-emerald-700 text-white px-3 py-1.5 rounded-lg font-medium">
          ✅ Tampilkan
        </button>

        <button id="btnExport"
                class="bg-blue-500 hover:bg-blue-600 text-white px-3 py-1.5 rounded-lg font-medium">
          📤 Export Excel
        </button>
      </div>
    </div>
  </section>

  <!-- TABEL REKAP (DI ATAS) -->
  <section class="bg-[#141d2f] shadow-xl rounded-2xl p-3 sm:p-4 border border-white/10">
    <h2 class="text-sm sm:text-base font-semibold text-gray-200 mb-2">
      📋 Tabel Rekap Kehadiran Per Kelas
    </h2>
    <div class="overflow-x-auto rounded-lg border border-white/10">
      <table class="min-w-full text-[11px] sm:text-xs text-center" id="rekapTable">
        <thead id="tableHeader" class="bg-[#243B55] text-white"></thead>
        <tbody id="tableBody" class="bg-[#020617]"></tbody>
      </table>
    </div>
  </section>

  <!-- GRAFIK-GRAFIK (DI BAWAH) -->
  <section class="grid grid-cols-1 md:grid-cols-2 gap-4">
    <div class="bg-[#141d2f] rounded-2xl p-3 sm:p-4 border border-white/10">
      <h3 class="text-sm font-semibold text-gray-200 mb-2">
        📈 Tren Persentase Kehadiran (Line Chart)
      </h3>
      <canvas id="lineChart" height="200"></canvas>
    </div>
    <div class="bg-[#141d2f] rounded-2xl p-3 sm:p-4 border border-white/10">
      <h3 class="text-sm font-semibold text-gray-200 mb-2">
        📊 Perbandingan Antar Kelas (Bar Chart)
      </h3>
      <canvas id="barChart" height="200"></canvas>
    </div>
  </section>
</main>

<!-- 🔄 Overlay Download Excel -->
<div id="downloadOverlay"
     class="hidden fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 text-center">
  <div class="flex flex-col items-center justify-center bg-[#141d2f]/90 rounded-2xl px-8 py-6 border border-white/10 shadow-2xl">
    <div class="loader mb-4 scale-110"></div>
    <p class="text-gray-200 font-medium text-base tracking-wide">Menyiapkan file Excel...</p>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
  // ===== URL PARAM UNTUK LINK LANGSUNG =====
  const urlParams       = new URLSearchParams(window.location.search);
  let urlFilterKelas    = urlParams.get('kelas');   // nama kelas persis, contoh: "XI RPL 1"
  let urlFilterTingkat  = urlParams.get('tingkat'); // contoh: "X", "XI", "XII"

  // ====== KONFIG API PUBLIK ======
  const REKAP_API_URL    = '/system-absent/public_api/rekap_statistik.php';
  const REKAP_EXPORT_URL = '/system-absent/public/export_rekap_statistik_excel.php';

  let lineChart = null;
  let barChart  = null;
  let rekapDataGlobal = {};
  let currentMonths   = [];
  let filterOptionsInitialized = false;

  function monthName(id) {
    const map = ['Jan','Feb','Mar','Apr','Mei','Jun','Jul','Agu','Sep','Okt','Nov','Des'];
    return map[id - 1] || '-';
  }

  function getMonthRange(start, end) {
    const s = new Date(start);
    const e = new Date(end);
    const months = [];
    if (isNaN(s) || isNaN(e)) return months;
    while (s <= e) {
      months.push(s.getMonth() + 1);
      s.setMonth(s.getMonth() + 1);
    }
    return months;
  }

  function renderTableHeader(months) {
    let header = `
      <tr>
        <th class="p-2 border bg-[#243B55] text-white text-center">KELAS</th>`;
    months.forEach(m => {
      header += `
        <th class="p-2 border bg-[#243B55] text-white text-center">${monthName(m)}</th>
        <th class="p-2 border bg-[#243B55] text-white text-center">${monthName(m)} (%)</th>`;
    });
    header += `
        <th class="p-2 border bg-[#243B55] text-white text-center">TOTAL HADIR</th>
        <th class="p-2 border bg-[#243B55] text-white text-center">RATA-RATA</th>
      </tr>`;
    document.getElementById('tableHeader').innerHTML = header;
  }

  // ============================
  // 🔹 RENDER TABEL + HITUNG RATA2 ANTAR KELAS
  // ============================
  function renderTableBody(data, months) {
    const bodyEl       = document.getElementById('tableBody');
    const avgAllEl     = document.getElementById('avgAll');
    const avgAllInfoEl = document.getElementById('avgAllInfo');

    let body = '';
    let totalKelasAvg = 0;
    let classCount    = 0;

    const kelasList = Object.keys(data);
    if (!kelasList.length) {
      bodyEl.innerHTML = `<tr>
        <td colspan="${months.length * 2 + 3}" class="p-3 text-gray-400">
          Tidak ada data untuk periode ini
        </td>
      </tr>`;
      if (avgAllEl)     avgAllEl.textContent = '0.0%';
      if (avgAllInfoEl) avgAllInfoEl.textContent = 'Tidak ada kelas pada periode ini.';
      return;
    }

    for (const kelas of kelasList) {
      const bulanArr = Array.isArray(data[kelas]) ? data[kelas] : [];

      let totalHadirKelas = 0;
      let sumPersen  = 0;
      let countPersen = 0;

      let row = `
        <tr class="hover:bg-[#1E2B44] transition">
          <td class="p-2 border font-semibold whitespace-nowrap">${kelas}</td>`;

      months.forEach(m => {
        const bulan  = bulanArr.find(b => b.bulan === m);
        const hadir  = bulan ? (bulan.hadir  ?? 0) : 0;
        const persen = bulan ? (bulan.persen ?? 0) : 0;

        totalHadirKelas += hadir;
        if (bulan) {
          sumPersen  += persen;
          countPersen++;
        }

        row += `
          <td class="p-2 border">${hadir}</td>
          <td class="p-2 border text-cyan-300">${persen.toFixed ? persen.toFixed(1) : persen}%</td>`;
      });

      const avgKelas = countPersen ? (sumPersen / countPersen) : 0;
      totalKelasAvg += avgKelas;
      classCount++;

      row += `
        <td class="p-2 border font-bold text-white">${totalHadirKelas}</td>
        <td class="p-2 border text-cyan-300">${avgKelas.toFixed(1)}%</td>
      </tr>`;

      body += row;
    }

    bodyEl.innerHTML = body;

    // 🔹 Rata-rata semua kelas = rata-rata dari rata-rata kelas
    const overallAvg = classCount ? (totalKelasAvg / classCount) : 0;
    if (avgAllEl) {
      avgAllEl.textContent = overallAvg.toFixed(1) + '%';
    }
    if (avgAllInfoEl) {
      avgAllInfoEl.textContent = `Berdasarkan ${classCount} kelas pada periode yang dipilih.`;
    }
  }

  function renderCharts(data, months) {
    const labels = months.map(monthName);
    const datasets = Object.keys(data).map((kelas, i) => {
      const colors = ['#3b82f6', '#ef4444', '#eab308', '#22c55e', '#8b5cf6', '#06b6d4'];
      const color  = colors[i % colors.length];
      const values = months.map(m => {
        const bulanData = (data[kelas] || []).find(b => b.bulan === m);
        return bulanData ? (bulanData.persen ?? 0) : 0;
      });
      return {
        label: kelas,
        borderColor: color,
        backgroundColor: color + '90',
        data: values
      };
    });

    const lineCtx = document.getElementById('lineChart');
    const barCtx  = document.getElementById('barChart');

    if (lineChart) lineChart.destroy();
    if (barChart)  barChart.destroy();

    if (lineCtx) {
      lineChart = new Chart(lineCtx, {
        type: 'line',
        data: { labels, datasets },
        options: {
          responsive: true,
          plugins: {
            legend: { labels: { color: '#e5e7eb', font: { size: 10 } } }
          },
          scales: {
            x: { ticks: { color: '#9ca3af' } },
            y: { ticks: { color: '#9ca3af' }, beginAtZero: true }
          }
        }
      });
    }

    if (barCtx) {
      barChart = new Chart(barCtx, {
        type: 'bar',
        data: { labels, datasets },
        options: {
          responsive: true,
          plugins: {
            legend: { labels: { color: '#e5e7eb', font: { size: 10 } } }
          },
          scales: {
            x: { ticks: { color: '#9ca3af' } },
            y: { ticks: { color: '#9ca3af' }, beginAtZero: true }
          }
        }
      });
    }
  }

  function initFilterOptions() {
    if (filterOptionsInitialized) return;
    const select = document.getElementById('filterTingkat');
    if (!select) return;

    const prefixes = new Set();
    Object.keys(rekapDataGlobal).forEach(kelas => {
      const prefix = (kelas || '').split(' ')[0];
      if (prefix) prefixes.add(prefix);
    });

    select.innerHTML = '<option value="semua">Semua kelas</option>';
    prefixes.forEach(prefix => {
      const opt = document.createElement('option');
      opt.value = prefix;
      opt.textContent = prefix;
      select.appendChild(opt);
    });

    if (urlFilterTingkat && prefixes.has(urlFilterTingkat)) {
      select.value = urlFilterTingkat;
    }

    filterOptionsInitialized = true;
  }

  function applyFilter() {
    const select = document.getElementById('filterTingkat');
    const infoEl = document.getElementById('urlFilterInfo'); // mungkin tidak ada, aman

    let filter = select ? select.value : 'semua';

    if (urlFilterTingkat && select) {
      filter       = urlFilterTingkat;
      select.value = urlFilterTingkat;
    }

    const filtered = {};
    const kelasKeys = Object.keys(rekapDataGlobal);

    if (urlFilterKelas && kelasKeys.includes(urlFilterKelas)) {
      filtered[urlFilterKelas] = rekapDataGlobal[urlFilterKelas];
      if (infoEl) {
        infoEl.textContent = `Menampilkan hanya kelas: ${urlFilterKelas} (dari parameter URL ?kelas=...)`;
        infoEl.classList.remove('hidden');
      }
    } else {
      kelasKeys.forEach(kelas => {
        const prefix = (kelas || '').split(' ')[0];
        if (filter === 'semua' || prefix === filter) {
          filtered[kelas] = rekapDataGlobal[kelas];
        }
      });

      if (infoEl) {
        if (urlFilterTingkat && filter === urlFilterTingkat) {
          infoEl.textContent = `Menampilkan hanya kelas tingkat ${urlFilterTingkat} (dari parameter URL ?tingkat=...)`;
          infoEl.classList.remove('hidden');
        } else {
          infoEl.classList.add('hidden');
        }
      }
    }

    renderTableBody(filtered, currentMonths);
    renderCharts(filtered, currentMonths);
  }

  async function loadData() {
    const periodeVal = document.getElementById('periode').value.split(',');
    const [start, end] = periodeVal;
    const months = getMonthRange(start, end);
    currentMonths = months;
    renderTableHeader(months);

    try {
      const res = await fetch(`${REKAP_API_URL}?start=${start}&end=${end}`);
      const json = await res.json();
      if (!json.success) throw new Error(json.error || 'Gagal memuat data');

      rekapDataGlobal = json.data || {};
      initFilterOptions();
      applyFilter();
    } catch (err) {
      document.getElementById('tableBody').innerHTML =
        `<tr><td colspan="20" class="text-red-400 p-3">⚠️ ${err.message}</td></tr>`;
      const avgAllEl     = document.getElementById('avgAll');
      const avgAllInfoEl = document.getElementById('avgAllInfo');
      if (avgAllEl)     avgAllEl.textContent = '0.0%';
      if (avgAllInfoEl) avgAllInfoEl.textContent = 'Gagal memuat data.';
    }
  }

  // SPINNER HALAMAN (fade in/out)
  window.addEventListener('load', () => {
    const spinner = document.getElementById('spinnerOverlay');
    const content = document.querySelector('.page-root');
    if (spinner && content) {
      content.style.opacity = '0';
      spinner.style.transition = 'opacity 0.6s ease';
      spinner.style.opacity = '0';
      setTimeout(() => {
        spinner.remove();
        content.style.transition = 'opacity 0.8s ease';
        content.style.opacity = '1';
      }, 650);
    }
  });

  // EXPORT EXCEL (tetap sama)
  document.getElementById('btnExport')?.addEventListener('click', async () => {
    const periode = document.getElementById('periode').value.split(',');
    const [start, end] = periode;
    const overlay = document.getElementById('downloadOverlay');
    overlay.classList.remove('hidden');

    try {
      const url = `${REKAP_EXPORT_URL}?start=${start}&end=${end}`;
      const res = await fetch(url);
      if (!res.ok) throw new Error('Gagal menyiapkan file.');

      const disposition = res.headers.get('Content-Disposition');
      let filename = 'Rekap_Statistik.xlsx';
      if (disposition && disposition.includes('filename=')) {
        filename = disposition.split('filename=')[1].replace(/["']/g, '');
      }

      const blob = await res.blob();
      const blobUrl = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = blobUrl;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();

      setTimeout(() => {
        window.URL.revokeObjectURL(blobUrl);
        overlay.classList.add('hidden');
      }, 800);
    } catch (err) {
      overlay.classList.add('hidden');
      alert('⚠️ Terjadi kesalahan: ' + err.message);
    }
  });

  // EVENT TOMBOL & FILTER
  document.getElementById('btnTampil')?.addEventListener('click', () => {
    loadData();
  });

  document.getElementById('filterTingkat')?.addEventListener('change', () => {
    urlFilterTingkat = null;
    urlFilterKelas   = null;
    applyFilter();
  });

  // LOAD PERTAMA
  window.addEventListener('DOMContentLoaded', () => {
    loadData();
  });
</script>



</body>
</html>
