<?php
// Fallback agar URL pendek & panjang dua-duanya bisa dipakai
if (!isset($_GET['token']) && preg_match('~/rekap/([A-Za-z0-9]+)$~', $_SERVER['REQUEST_URI'], $m)) {
    $_GET['token'] = $m[1];
}

require_once __DIR__ . '/../inc/db.php';
$token = $_GET['token'] ?? '';
?>
<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <link rel="icon" type="image/png" href="/system-absent/assets/icons/logo.png">
  <title>Rekap Absensi</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen flex flex-col items-center justify-center p-3">

<div id="main" class="w-full max-w-3xl mx-auto bg-white shadow-lg rounded-2xl p-4">
  <header class="text-center mb-4">
    <h1 class="text-xl font-bold text-gray-800">Rekap Kehadiran</h1>
    <p id="periode" class="text-sm text-gray-500"></p>
  </header>

  <!-- === Pilih Kelas === -->
  <section id="classSection" class="mb-4 hidden">
    <label class="block text-gray-700 font-medium mb-1">Pilih Kelas</label>
    <select id="classSelect" class="w-full p-2 border rounded-lg focus:ring-2 focus:ring-blue-400">
      <option value="">-- Pilih Kelas --</option>
    </select>
  </section>

  <!-- === Dropdown Download + Tombol Statistik === -->
  <div class="flex justify-end items-center gap-2 mb-4">
    <!-- Tombol Statistik -->
    <button 
      onclick="window.open('https://ideaproper.my.id/system-absent/public/rekap-statistik.php', '_blank')"
      class="flex items-center gap-2 bg-gradient-to-r from-purple-500 to-indigo-500 hover:opacity-90 text-white font-semibold px-4 py-2 rounded-xl shadow-md transition">
      <svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
              d="M11 3v18M4 10h7m0 0h7m-7 0V3m7 7h3m-3 0v10m0 0H4m0 0v-4"/>
      </svg>
      Statistik
    </button>

    <!-- Tombol Download -->
    <div id="downloadSection" class="hidden relative">
      <button id="downloadDropdownBtn" 
              class="flex items-center gap-2 bg-gradient-to-r from-blue-500 to-cyan-400 hover:opacity-90 text-white font-semibold px-4 py-2 rounded-xl shadow-md transition">
        <svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                d="M4 16v2a2 2 0 002 2h12a2 2 0 002-2v-2M7 10l5 5m0 0l5-5m-5 5V4" />
        </svg>
        <span>Download</span>
        <svg xmlns="http://www.w3.org/2000/svg" class="w-4 h-4 ml-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                d="M19 9l-7 7-7-7" />
        </svg>
      </button>

      <!-- Menu Dropdown -->
      <div id="downloadMenu" 
           class="hidden absolute z-10 mt-2 w-44 bg-white border border-gray-200 rounded-xl shadow-lg overflow-hidden transition-all duration-200">
        <a href="#" onclick="downloadFile('excel')" 
           class="flex items-center gap-2 px-4 py-2 text-gray-700 hover:bg-gray-100 transition">
         <span>Download Excel</span>
        </a>
      </div>
    </div>
  </div>

  <!-- === Filter Status + Kegiatan === -->
  <div id="filterWrapper" class="hidden mb-3 flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2 text-sm">
    <div>
      <label class="text-gray-500 mr-2">Filter Status:</label>
      <select id="filterStatus" 
              class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1 text-sm focus:ring-2 focus:ring-blue-400 transition">
        <option value="">Semua</option>
        <option value="Hadir">Hadir</option>
        <option value="Izin">Izin</option>
        <option value="Sakit">Sakit</option>
        <option value="Alfa">Alfa</option>
      </select>
    </div>
    <div>
      <label class="text-gray-500 mr-2">Filter Kegiatan:</label>
      <select id="filterActivity"
              class="bg-[#1E2B44] border border-white/10 text-white rounded-lg px-3 py-1 text-sm focus:ring-2 focus:ring-blue-400 transition">
        <option value="">Semua</option>
      </select>
    </div>
  </div>

  <!-- === Container Rekap Per Tanggal === -->
  <div id="rekapContainer" class="hidden fade space-y-6"></div>

  <footer class="text-center text-xs text-gray-500 mt-4">
    Link aktif hingga: <span id="expired"></span>
  </footer>
</div>

<script>
// === Ambil Token dari URL ===
const token = (() => {
  let t = new URLSearchParams(window.location.search).get('token');
  if (!t) {
    const m = window.location.pathname.match(/rekap\/([A-Za-z0-9]+)/);
    if (m) t = m[1];
  }
  return t;
})();

if (!token) {
  document.body.innerHTML = "<p class='text-center text-red-500 mt-10'>⚠️ Token tidak ditemukan.</p>";
}

// === Style tambahan (shimmer, spinner, progress) ===
const style = document.createElement('style');
style.textContent = `
@keyframes shimmer { 0% { background-position: -400px 0; } 100% { background-position: 400px 0; } }
@keyframes spin { to { transform: rotate(360deg); } }
.spinner { border: 4px solid #e5e7eb; border-top-color: #3b82f6; border-radius: 50%; width: 28px; height: 28px; animation: spin .7s linear infinite; }
.fade { transition: opacity .4s ease, transform .4s ease; }
`;
document.head.appendChild(style);

// === Ambil daftar kelas ===
let usePeriod = 1; // default, biar bisa dibaca di luar fetch()
fetch(`/system-absent/public_api/rekap_public.php?token=${token}`)
  .then(r => r.json())
  .then(res => {
    usePeriod = (res.periode && typeof res.periode.use_period !== "undefined") 
      ? Number(res.periode.use_period) 
      : 1;
    if (!res.success) throw new Error(res.error || 'Gagal memuat data');

    // === Tampilkan periode atau dropdown bulan/tahun ===
    if (usePeriod === 1 && res.periode && res.periode.start && res.periode.end) {
      document.getElementById('periode').textContent = 
        `Periode ${res.periode.start} s/d ${res.periode.end}`;
    } else {
      document.getElementById('periode').innerHTML = `
        <div class="flex justify-center items-center gap-2 text-sm text-gray-700">
          <label>Bulan:</label>
          <select id="monthSelect" class="border rounded-md px-2 py-1">
            ${Array.from({ length: 12 }, (_, i) => {
              const m = i + 1;
              const nama = new Date(2025, i).toLocaleString('id-ID', { month: 'long' });
              return `<option value="${m}">${nama}</option>`;
            }).join('')}
          </select>
          <label>Tahun:</label>
          <select id="yearSelect" class="border rounded-md px-2 py-1">
            ${[2023, 2024, 2025, 2026].map(y => 
              `<option value="${y}" ${y === new Date().getFullYear() ? 'selected' : ''}>${y}</option>`
            ).join('')}
          </select>
        </div>`;
    }

    document.getElementById('expired').textContent = 
      new Date(res.periode.expired_at).toLocaleString('id-ID');

    // === Isi dropdown kelas ===
    const select = document.getElementById('classSelect');
    res.kelas.forEach(k => {
      const opt = document.createElement('option');
      opt.value = k.id;
      opt.textContent = k.class_name;
      select.appendChild(opt);
    });

    document.getElementById('classSection').classList.remove('hidden');
    select.addEventListener('change', () => {
      if (select.value) {
        loadRekap(select.value);
        document.getElementById('downloadSection').classList.remove('hidden');
      } else {
        document.getElementById('downloadSection').classList.add('hidden');
      }
    });

    // ✅ Listener bulan/tahun agar auto reload
    if (usePeriod === 0) {
      const monthSel = document.getElementById("monthSelect");
      const yearSel = document.getElementById("yearSelect");

      function reloadIfReady() {
        if (select.value) {
          const container = document.getElementById("rekapContainer");
          container.innerHTML = `
            <div class="text-center py-6 text-gray-400">
              <div class="spinner mx-auto mb-2"></div>
              <p>Sedang memuat ulang data...</p>
            </div>`;
          container.style.opacity = "0.5";
          
          setTimeout(() => {
            loadRekap(select.value);
            setTimeout(() => (container.style.opacity = "1"), 400);
          }, 300);
        }
      }

      monthSel.addEventListener("change", reloadIfReady);
      yearSel.addEventListener("change", reloadIfReady);
    }

  })
  .catch(err => {
    document.body.innerHTML = `<p class='text-center text-red-500 mt-10'>⚠️ ${err.message}</p>`;
  });

// === Smooth Load Rekap ===
function loadRekap(classId) {
  const container = document.getElementById('rekapContainer');
  container.innerHTML = `
    <div class="text-center py-6 text-gray-400">
      <div class="spinner mx-auto mb-2"></div>
      <p>Memuat rekap per tanggal...</p>
    </div>`;
  container.classList.remove('hidden');
  container.style.opacity = '0.3';

  let url = `/system-absent/public_api/rekap_public.php?token=${token}&class_id=${classId}`;
  if (typeof usePeriod !== "undefined" && usePeriod === 0) {
    const bulan = document.getElementById("monthSelect").value;
    const tahun = document.getElementById("yearSelect").value;
    url += `&bulan=${bulan}&tahun=${tahun}`;
  }

  fetch(url)
    .then(r => r.json())
    .then(res => {
      if (!res.success) throw new Error(res.error || 'Gagal memuat data');
      const rekap = res.rekap;
      let html = '';

      Object.entries(rekap).forEach(([tgl, data]) => {
        const tanggalLabel = new Date(tgl).toLocaleDateString('id-ID', { 
          weekday: 'long', day: 'numeric', month: 'long', year: 'numeric' 
        });

        const totalHadir = data.filter(s => s.status === 'Hadir').length;

        // === Tentukan label kegiatan per tanggal ===
        const uniqueActs = Array.from(new Set(
          data.map(s => s.activity_name).filter(a => a && a !== '')
        ));
        let activityLabel = '-';
        if (uniqueActs.length === 1) {
          activityLabel = uniqueActs[0];
        } else if (uniqueActs.length > 1) {
          activityLabel = 'Beragam kegiatan';
        }

        html += `
  <div class="border rounded-xl overflow-hidden shadow-sm transition hover:shadow-md">
    <div class="bg-blue-50 px-3 py-2 font-semibold text-blue-700 border-b flex flex-col sm:flex-row sm:items-center sm:justify-between gap-1 sm:gap-2">
      <div class="flex items-center gap-1">
        <span>📅 ${tanggalLabel}</span>
      </div>
      <div class="text-xs">
        <span class="inline-flex items-center px-3 py-1 rounded-full bg-blue-100 text-blue-800 border border-blue-200">
          ${activityLabel}
        </span>
      </div>
      <div class="text-xs text-gray-500 text-left sm:text-right">
        <span class="text-green-600 font-semibold">${totalHadir}</span> hadir dari 
        <span class="font-medium">${data.length}</span> siswa
      </div>
    </div>
    <table class="w-full text-sm border-collapse">
      <thead class="bg-blue-500 text-white">
        <tr>
          <th class="p-2 border">Nama</th>
          <th class="p-2 border">Status</th>
        </tr>
      </thead>
      <tbody>
        ${data.map(s => {
          const status = s.status || '-';
          const statusLower = status.toLowerCase();
          const activityLower = (s.activity_name || '').toLowerCase();
          const statusClass =
            status === 'Hadir' ? 'text-green-600 font-semibold' :
            status === 'Izin'  ? 'text-blue-600'  :
            status === 'Sakit' ? 'text-yellow-600':
            status === 'Alfa'  ? 'text-red-600'   :
                                 'text-gray-400';

          return `
            <tr class="border-b hover:bg-gray-50 transition-all duration-200"
                data-status="${statusLower}"
                data-activity="${activityLower}">
              <td class="p-2 text-left">${s.name}</td>
              <td class="p-2 ${statusClass}">${status}</td>
            </tr>`;
        }).join('')}
      </tbody>
    </table>
  </div>`;
      });

      if (!html) html = `<p class="text-center text-gray-400 py-4">Tidak ada data absensi untuk kelas ini.</p>`;
      container.innerHTML = html;
      setTimeout(() => container.style.opacity = '1', 200);
      document.getElementById('filterWrapper').classList.remove('hidden');

      // build ulang daftar kegiatan untuk filter
      buildActivityFilterFromDOM();
      // apply filter terakhir (kalau user sudah pilih sesuatu sebelumnya)
      applyFilters();
    })
    .catch(err => {
      container.innerHTML = `<p class="text-center text-red-500 py-4">⚠️ ${err.message}</p>`;
      container.style.opacity = '1';
    });
}

// === Bangun isi dropdown kegiatan dari DOM ===
function buildActivityFilterFromDOM() {
  const select = document.getElementById('filterActivity');
  if (!select) return;

  const rows = document.querySelectorAll('#rekapContainer tbody tr');
  const set = new Set();

  rows.forEach(row => {
    const act = (row.dataset.activity || '').trim();
    if (act) set.add(act);
  });

  const current = select.value;
  select.innerHTML = '<option value="">Semua</option>';

  Array.from(set).sort((a, b) => a.localeCompare(b, 'id-ID')).forEach(act => {
    const opt = document.createElement('option');
    opt.value = act;
    // tampilkan apa adanya (karena nama dari master data sudah rapi)
    opt.textContent = act;
    select.appendChild(opt);
  });

  if (current && [...select.options].some(o => o.value === current)) {
    select.value = current;
  } else {
    select.value = '';
  }
}

// === Dropdown Menu Download ===
const btn = document.getElementById('downloadDropdownBtn');
const menu = document.getElementById('downloadMenu');
btn.addEventListener('click', () => menu.classList.toggle('hidden'));
window.addEventListener('click', (e) => {
  if (!btn.contains(e.target) && !menu.contains(e.target)) menu.classList.add('hidden');
});

// === Overlay Loading Utility ===
function showOverlay(msg = "Menyiapkan file...") {
  const overlay = document.createElement('div');
  overlay.id = "overlayLoading";
  overlay.className = 'fixed inset-0 bg-black/40 flex flex-col items-center justify-center z-50 text-center';
  overlay.innerHTML = `
    <div class="bg-white rounded-xl px-6 py-5 shadow-lg">
      <div class="spinner mx-auto mb-3"></div>
      <p class="text-gray-700 font-medium mb-2">${msg}</p>
      <div id="progressBar" class="h-1 bg-blue-500 w-0 rounded transition-all duration-500"></div>
    </div>`;
  document.body.appendChild(overlay);

  const bar = overlay.querySelector('#progressBar');
  bar.style.width = '25%';
  setTimeout(() => (bar.style.width = '60%'), 700);
  setTimeout(() => (bar.style.width = '90%'), 1500);
}

function hideOverlay() {
  const o = document.getElementById('overlayLoading');
  if (o) {
    o.style.opacity = '0';
    setTimeout(() => o.remove(), 400);
  }
}

// === Download File (Excel) ===
async function downloadFile() {
  const classSelect = document.getElementById('classSelect');
  const classId = classSelect.value || '';
  let url = `/system-absent/public/export_rekap_excel.php?token=${token}`;

  if (classId) url += `&class_id=${classId}`;

  if (typeof usePeriod !== "undefined" && usePeriod === 0) {
    const bulan = document.getElementById("monthSelect")?.value;
    const tahun = document.getElementById("yearSelect")?.value;
    if (bulan && tahun) {
      url += `&bulan=${bulan}&tahun=${tahun}`;
    }
  }

  showOverlay('⏳ Menyiapkan file Excel...');

  try {
    const res = await fetch(url);
    if (!res.ok) throw new Error('Gagal menyiapkan file.');

    const disposition = res.headers.get('Content-Disposition');
    let filename = 'rekap_absensi.xlsx';
    if (disposition && disposition.includes('filename=')) {
      filename = disposition.split('filename=')[1].replace(/["']/g, '');
    }

    const blob = await res.blob();
    const blobUrl = window.URL.createObjectURL(blob);

    const a = document.createElement('a');
    a.href = blobUrl;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();

    setTimeout(() => {
      window.URL.revokeObjectURL(blobUrl);
      hideOverlay();
    }, 600);
  } catch (err) {
    alert('⚠️ Terjadi kesalahan: ' + err.message);
    hideOverlay();
  }
}

// === FILTER: Status + Kegiatan ===
const filterStatus  = document.getElementById("filterStatus");
const filterActivity = document.getElementById("filterActivity");

function applyFilters() {
  const statusVal   = (filterStatus.value || '').toLowerCase();
  const activityVal = (filterActivity.value || '').toLowerCase();
  const rows = document.querySelectorAll("#rekapContainer table tbody tr");

  let count = 0;
  rows.forEach(row => {
    const rowStatus = (row.dataset.status || '').toLowerCase();
    const rowAct    = (row.dataset.activity || '').toLowerCase();

    const statusMatch   = !statusVal   || rowStatus === statusVal;
    const activityMatch = !activityVal || rowAct    === activityVal;

    if (statusMatch && activityMatch) {
      row.classList.remove("hidden");
      count++;
    } else {
      row.classList.add("hidden");
    }
  });

  // Sembunyikan card tanggal yang isinya kosong semua
  document.querySelectorAll('#rekapContainer > div').forEach(card => {
    const visibleRows = card.querySelectorAll('tbody tr:not(.hidden)').length;
    if (visibleRows === 0) {
      card.classList.add('hidden');
    } else {
      card.classList.remove('hidden');
    }
  });

  console.log(`Menampilkan ${count} baris dengan status=${statusVal || "semua"} dan kegiatan=${activityVal || "semua"}`);
}

filterStatus.addEventListener("change", applyFilters);
filterActivity.addEventListener("change", applyFilters);

</script>

</body>
</html>
